// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n.swing;

import de.caff.annotation.NotNull;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  Menu with localized label.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RJMenu
  extends    JMenu
  implements Localizable
{
  private static final long serialVersionUID = 6209438987013187485L;
  protected String resTag;         // name of the resource of this menu's name

  /**
   *  @param  tag       resource tag for the item
   */
  public RJMenu(@NotNull String tag) {
    this(tag, false);
  }

  /**
   *  @param  tag       resource tag for the item
   *  @param  tearoff   tearoff menu?
   */
  public RJMenu(@NotNull String tag, boolean tearoff) {
    super("", tearoff);
    super.setLocale(I18n.getDefaultLocale());
    setTag(tag);
  }

  /**
   * @param  tag       resource tag for the item
   * @param  entries   initial menu actions, use {@code null} for separator
   */
  public RJMenu(@NotNull String tag, Action ... entries) {
    this(tag, false, entries);
  }

  /**
   * @param  tag       resource tag for the item
   * @param  tearoff   tearoff menu?
   * @param  entries   initial menu actions, use {@code null} for separator
   */
  public RJMenu(@NotNull String tag, boolean tearoff, Action ... entries) {
    super("", tearoff);
    super.setLocale(I18n.getDefaultLocale());
    setTag(tag);
    for (Action entry : entries) {
      if (entry == null) {
        addSeparator();
      }
      else {
        add(entry);
      }
    }
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides {@code JComponent.removeNotify} to check if
   * this button is currently set as the default button on the
   * {@code RootPane}, and if so, sets the {@code RootPane}'s
   * default button to {@code null} to ensure the
   * {@code RootPane} doesn't hold onto an invalid button reference.
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   *  Change the tag of this item.
   *  @param  tag       new resource tag 
   */
  public void setTag(@NotNull String tag) {
    resTag = tag;
    I18nHelper.setProperties(this, tag, getLocale());
  }

  /**
   *  Change the locale.
   *  @param  l    new locale
   */
  @Override
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      if (resTag != null) {
        I18nHelper.setProperties(this, resTag, getLocale());
      }
    }
  }

  /**
   *  Get the locale. Instead of throwing an {@code IllegalComponentStateException}
   *  when there's no locale we just return null.
   *  @return the Locale of this Label or null
   */
  @Override
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }

  @Override
  public void updateUI()
  {
    super.updateUI();
    for (int i = getItemCount() - 1;  i >= 0;  --i) {
      final JMenuItem item = getItem(i);
      if (item != null) {
        item.updateUI();
      }
    }
  }

  /**
   *  Overwritten to set the locale of the JMenuItem when unset.
   *  @param  m   menu item to add
   */
  @Override
  public JMenuItem add(@NotNull JMenuItem m) {
    if (m instanceof Localizable) {
      Localizable l = (Localizable)m;
      if (l.getLocale() == null) {
        l.setLocale(getLocale());
      }
    }
    return super.add(m);
  }

  /**
   *  Overwritten to set the locale of the JMenuItem when unset.
   *  @param  m   menu item to add
   */
  @Override
  public JMenuItem insert(@NotNull JMenuItem m, int index) {
    if (m instanceof Localizable) {
      Localizable l = (Localizable)m;
      if (l.getLocale() == null) {
        l.setLocale(getLocale());
      }
    }
    return super.insert(m, index);
  }
}

