// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  A JLabel with localized resource.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RJLabel 
  extends    JLabel 
  implements Localizable 
{
  private static final long serialVersionUID = -8023534734652838087L;
  /** Resource tag for label text. */
  protected String   textResource;  
  /** Extra text (unresourced) to add. */
  protected String   addText;     

  /**
   *  Initialize method.
   *  @param  textRes  resource tag of label text
   *  @param  add      unresourced tag to add
   */
  private void init(@NotNull String textRes,
                    @Nullable String add)
  {
    super.setLocale(I18n.getDefaultLocale());
    I18nHelper.setProperties(this, textRes, getLocale());
    if (add != null) {
      setText(getText()+add);
    }
    textResource = textRes;
    addText      = add;
  }

  /**
   *  Constructor.
   *  @param   textRes    resource tag of the label text
   */
  public RJLabel(@NotNull String textRes)
  {
    init(textRes, null);
  }

  /**
   *  Constructor.
   *  @param   textRes    resource tag of the label text
   *  @param   align      horizontal alignment
   */
  public RJLabel(@NotNull String textRes, int align)
  {
    init(textRes, null);
    setHorizontalAlignment(align);
  }

  /**
   *  Constructor.
   *  @param   textRes    resource tag of the label text
   *  @param   add        text to add
   */
  public RJLabel(@NotNull String textRes, String add)
  {
    init(textRes, add);
  }

  /**
   *  Constructor.
   *  @param   textRes    resource tag of the label text
   *  @param   add        text to add
   *  @param   align      horizontal alignment
   */
  public RJLabel(@NotNull String textRes, String add, int align)
  {
    init(textRes, add);
    setHorizontalAlignment(align);
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides {@code JComponent.removeNotify} to check if
   * this button is currently set as the default button on the
   * {@code RootPane}, and if so, sets the {@code RootPane}'s
   * default button to {@code null} to ensure the
   * {@code RootPane} doesn't hold onto an invalid button reference.
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   *  Change the locale.
   *  @param  l   new locale
   */
  @Override
  public void setLocale(Locale l)
  {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      if (textResource != null) {
        I18nHelper.setProperties(this, textResource, getLocale());
        if (addText != null) {
          setText(getText()+addText);
        }
        invalidate();
      }
    }
  }

  /**
   *  Get the locale. Instead of throwing 
   *  an {@code IllegalComponentStateException}
   *  when there's no locale we just return null.
   *  @return the Locale of this Label or null
   */
  @Override
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }
}

