// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;
import de.caff.util.settings.BoundsPreferenceProperty;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.util.Locale;

/**
 *  JFrame which notifies it's components of locale changes.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RJFrame 
  extends    JFrame 
  implements Localizable 
{
  private static final long serialVersionUID = 4970577955076794806L;
  protected String titleRes = null;      // resource for title

  /**
   *
   */
  public RJFrame() {
    super.setLocale(I18n.getDefaultLocale());
  }

  /**
   *  @param  titleRes  resource for title
   *  @param  l         locale 
   */
  public RJFrame(String titleRes, Locale l) {
    super(I18n.getString(titleRes, l));
    setLocale(l);
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides {@code JComponent.removeNotify} to check if
   * this button is currently set as the default button on the
   * {@code RootPane}, and if so, sets the {@code RootPane}'s
   * default button to {@code null} to ensure the
   * {@code RootPane} doesn't hold onto an invalid button reference.
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   *  Change the title resource tag.
   *  @param  tag       new resource tag for the name
   */
  public void setTitleTag(String tag) {
    titleRes = tag;
    if (tag != null) {
      setTitle(I18n.getString(tag, getLocale()));
    }
  }

  /**
   *  Change the locale.
   *  @param  l   new locale
   */
  @Override
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      if (titleRes != null) {
        setTitle(I18n.getString(titleRes, l));
      }
    }
  }

  /**
   *  Get the current locale. Don't throws {@code IllegalComponentStateException}.
   *  @return current locale or null if unknown
   */
  @Override
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }

  /**
   * Setup usage of a preference property for keeping the size and position of the frame.
   * This should only be called once at program start.
   * @param boundsPreferenceProperty preference property used to keep the bounds
   * @param defaultSize default size used, if {@code null} the frame is set to maximal size
   */
  public void setupPreferences(@NotNull final BoundsPreferenceProperty boundsPreferenceProperty,
                               @Nullable final Dimension defaultSize)
  {
    if (!boundsPreferenceProperty.setWindowBounds(this)) {
      if (defaultSize == null) {
        // maximize
        final Toolkit toolkit = getToolkit();
        final Dimension screenSize = toolkit.getScreenSize();
        final Insets insets = toolkit.getScreenInsets(getGraphicsConfiguration());

        setBounds(insets.left, insets.top,
                  screenSize.width-insets.left-insets.right,
                  screenSize.height-insets.top-insets.bottom);
      }
      else {
        setSize(defaultSize);
      }
    }
    addComponentListener(new ComponentAdapter()
    {
      @Override
      public void componentResized(ComponentEvent e)
      {
        boundsPreferenceProperty.setBounds(RJFrame.this);
      }

      @Override
      public void componentMoved(ComponentEvent e)
      {
        boundsPreferenceProperty.setBounds(RJFrame.this);
      }
    });
  }

}
