// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n.swing;

import de.caff.generics.Empty;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  A combo box with resource.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
@SuppressWarnings("rawtypes") // necessary for Java 7
public class RJComboBox 
  extends    JComboBox 
  implements Localizable 
{
  private static final long serialVersionUID = 6097292415335683297L;
  protected boolean   localeUnknown = true;          // flag whether Locale is already known
  protected String[]  resTags = Empty.STRING_ARRAY;  // resource tags for items

  /**
   *  Constructs a new RJComboBox.
   */
  public RJComboBox() {
    super.setLocale(I18n.getDefaultLocale());
    setEditable(false);
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides {@code JComponent.removeNotify} to check if
   * this button is currently set as the default button on the
   * {@code RootPane}, and if so, sets the {@code RootPane}'s
   * default button to {@code null} to ensure the
   * {@code RootPane} doesn't hold onto an invalid button reference.
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   *  Adds an resourced item to this choice.
   *  @param  tag   resource tag
   */
  public void add(String tag) {
    addItemTag(tag);
  }

  /**
   *  Adds an resourced item to this choice.
   *  @param  tag   resource tag
   */
  @SuppressWarnings("unchecked") // necessary for Java 7
  public void addItemTag(String tag) {
    // --- add to Choice ---
    super.addItem(I18n.getString(tag, getLocale()));

    // --- add resource ---
    String[] newTags = new String[resTags.length+1];
    if (resTags.length > 0) {
      System.arraycopy(resTags, 0, newTags, 0, resTags.length);
    }
    newTags[resTags.length] = tag;
    resTags = newTags;
  }

  /**
   *  Get the resource of the specified index.
   *  @param  index   the index
   *  @return resource tag of the item belonging to index
   */
  public String getTag(int index) {
    return resTags[index];
  }

  /**
   *  Get the resource of the selected item.
   *  @return resource tag of the selected item
   */
  public String getSelectedTag() {
    return resTags[getSelectedIndex()];
  }

  /**
   *  Inserts the resourced item into this choice at the specified position.
   *  @param  tag   resource tag
   *  @param  index position to insert
   */
  @SuppressWarnings("unchecked") // necessary for Java 7
  public void insertItemAt(String tag, int index) {
    // --- insert in Choice ---
    super.insertItemAt(I18n.getString(tag, getLocale()), index);

    // --- insert resource ---
    String[] newTags = new String[resTags.length+1];
    if (index > 0) {
      System.arraycopy(resTags, 0, newTags, 0, index);
    }
    newTags[index] = tag;
    if (index < resTags.length) {
      System.arraycopy(resTags, index, newTags, index+1, resTags.length-index);
    }
    resTags = newTags;
  }

  /**
   *  Removes the indexed item from the choice.
   *  @param  index  index to remove
   */
  @Override
  public void remove(int index) {
    // --- remove in Choice ---
    super.remove(index);

    // --- remove in resource ---
    String[] newTags = new String[resTags.length-1];
    if (index > 0) {
      System.arraycopy(resTags, 0, newTags, 0, index);
    }
    if (index < resTags.length-1) {
      System.arraycopy(resTags, index+1, newTags, index, resTags.length-index-1);
    }
    resTags = newTags;
  }

  /**
   *  Removes the first occurence of given resource tag.
   *  @param  tag   tag to remove
   */
  public void remove(String tag) {
    // --- find tag ...
    for (int i = 0;   i < resTags.length;   i++) {
      if (resTags[i].equals(tag)) {
	// ... and remove ---
	remove(i);
	return;
      }
    }
  }

  /**
   *  Removes all items.
   */
  @Override
  public void removeAll() {
    super.removeAll();
    resTags = Empty.STRING_ARRAY;
  }

  /**
   *  Select the resourced item.
   *  @param  tag   resource tag of item
   */
  public void setSelectedIndex(String tag) {
    // --- find tag ...
    for (int i = 0;   i < resTags.length;   i++) {
      if (resTags[i].equals(tag)) {
	// ... and select ---
	setSelectedIndex(i);
	return;
      }
    }
  }

  /**
   *  Change the locale.
   *  @param  l   new locale
   */
  @SuppressWarnings("unchecked") // necessary for Java 7
  @Override
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);

    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      int selected = getSelectedIndex();
      super.removeAll();
      for (String resTag : resTags) {
        super.addItem(I18n.getString(resTag, getLocale()));
      }
      setSelectedIndex(selected);
    
      invalidate();
    }
  }

  /**
   *  Get the locale. Instead of throwing an {@code IllegalComponentStateException}
   *  when there's no locale we just return null.
   *  @return the Locale of this Label or null
   */
  @Override
  public Locale getLocale() {
    try {
      localeUnknown = false;
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      localeUnknown = true;
      return null;
    }
  }

  /**
   *  If we paint without knowing the locale we try to get and set it.
   *  @param  g   graphics context
   */
  @Override
  public void update(Graphics g) {
    if (localeUnknown) {
      setLocale(getLocale());
      // no need to paint
    }
    else {
      super.update(g);
    }
  }
}
