// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n.swing;

import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  Button with localized label.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RJButton
  extends    JButton 
  implements Localizable 
{
  private static final long serialVersionUID = -7937411737061904897L;
  /** Basic resource tag for button properties. */
  protected String   resTag;

  /**
   *  @param  tag       basic resource tag for the item
   */
  public RJButton(String tag)
  {
    super.setLocale(I18n.getDefaultLocale());
    setTag(tag);
  }

  protected void setTag(String baseTag)
  {
    resTag = baseTag;
    I18nHelper.setProperties(this, baseTag, getLocale());
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides {@code JComponent.removeNotify} to check if
   * this button is currently set as the default button on the
   * {@code RootPane}, and if so, sets the {@code RootPane}'s
   * default button to {@code null} to ensure the
   * {@code RootPane} doesn't hold onto an invalid button reference.
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   *  Change the locale of this Button
   *  @param  l   new Locale
   */
  @Override
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      if (resTag != null) {
        setTag(resTag);
        invalidate();
      }
    }
  }

  /**
   *  Get the locale. Instead of throwing an {@code IllegalComponentStateException}
   *  when there's no locale we just return null.
   *  @return the Locale of this Label or null
   */
  @Override
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }

}

