// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.i18n.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.util.swing.SwingHelper;
import de.caff.vic.RecolorableIcon;
import de.caff.vic.RecoloringVisitor;

import javax.swing.*;
import java.util.Locale;
import java.util.MissingResourceException;

/**
 *  Utility class for i18n settings.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class I18nHelper
{
  /**
   *  Set the relevant properties of a menu item.
   *  @param menuItem  menu item which properties are set
   *  @param tag       base i18n tag
   *  @param locale    locale to use
   */
  public static void setProperties(@NotNull JMenuItem menuItem, @NotNull String tag, @Nullable Locale locale)
  {
    setProperties((AbstractButton)menuItem, tag, locale);
    try {
      menuItem.setAccelerator(KeyStroke.getKeyStroke(I18n.getString(tag+I18n.SUFFIX_ACCELERATOR, locale)));
    } catch (MissingResourceException ignored) {
    }
  }

  /**
   *  Set the relevant properties of a button.
   *  @param button  button which properties are set
   *  @param tag     base i18n tag
   *  @param locale  locale to use
   */
  public static void setProperties(@NotNull AbstractButton button,
                                   @NotNull String tag,
                                   @Nullable Locale locale)
  {
    button.setText(I18n.getString(tag+I18n.SUFFIX_TEXT, locale));
    try {
      button.setToolTipText(I18n.getString(tag+I18n.SUFFIX_TOOLTIP, locale));
    } catch (MissingResourceException ignored) {
    }
    try {
      button.setMnemonic(I18n.getString(tag+I18n.SUFFIX_MNEMONIC, locale).charAt(0));
    } catch (Exception ignored) {
    }
    try {
      Icon icon = SwingHelper.loadIconResource(I18n.getString(tag+I18n.SUFFIX_ICON, locale));
      if (icon != null) {
        button.setIcon(icon);
        if (icon instanceof RecolorableIcon) {
          button.setDisabledIcon(((RecolorableIcon)icon).getRecoloredIcon(RecoloringVisitor.TO_LIGHTER_GRAY));
        }
      }
      else {
        button.setIcon(null);
      }
    } catch (MissingResourceException e) {
      button.setIcon(null);
    }
    try {
      Icon icon = SwingHelper.loadIconResource(I18n.getString(tag+I18n.SUFFIX_DISABLED_ICON, locale));
      if (icon != null) {
        button.setDisabledIcon(icon);
      }
    } catch (MissingResourceException ignored) {
    }

  }

  /**
   *  Set the relevant properties of a label.
   *  @param button  label which properties are set
   *  @param tag     base i18n tag
   *  @param locale  locale to use
   */
  public static void setProperties(@NotNull JLabel button,
                                   @NotNull String tag,
                                   @Nullable Locale locale)
  {
    button.setText(I18n.getString(tag+I18n.SUFFIX_TEXT, locale));
    try {
      button.setToolTipText(I18n.getString(tag+I18n.SUFFIX_TOOLTIP, locale));
    } catch (MissingResourceException e) {
    }
  }

}
