// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;
import de.caff.i18n.WeakReferencedLocalizable;
import de.caff.util.swing.SwingHelper;

import javax.swing.*;
import java.util.Locale;
import java.util.MissingResourceException;

/**
 *  A visible state is a state with a possible label,
 *  different icons for different states and a possible tooltip text.
 *
 *  @author  <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class VisibleState
        extends AbstractState
        implements Localizable
{
  /** Property name for label resource key. */
  private static final String LABEL_RESOURCE_KEY_PROPERTY = "LABEL_RESOURCE_KEY";
  /** Property name for tooltip resource key. */
  private static final String TOOLTIP_RESOURCE_KEY_PROPERTY = "TOOLTIP_RESOURCE_KEY";

  /** The current locale. */
  private Locale locale;

  /**
   *  Create a visible state.
   *  @param labelResourceKey resource key for labels and menu items
   *  @param tooltipResourceKey resource key for tooltips
   *  @param activeIcon icon to be displayed when state is active
   *  @param inactiveIcon icon to be displayed when state is inactive
   *  @param disabledIcon icon to be displayed when state is disabled
   *         (if {@code null} the Swing standard is used)
   */
  public VisibleState(String       labelResourceKey,
                      String       tooltipResourceKey,
                      Icon         activeIcon,
                      Icon         inactiveIcon,
                      Icon         disabledIcon)
  {
    init(labelResourceKey, tooltipResourceKey, activeIcon, inactiveIcon, disabledIcon);
  }

  /**
   * Create a new abstract state.
   */
  public VisibleState()
  {
    super();
  }

  /**
   *  Internal initialization.
   *  @param labelResourceKey resource key for labels and menu items
   *  @param tooltipResourceKey resource key for tooltips
   *  @param activeIcon icon to be displayed when state is active
   *  @param inactiveIcon icon to be displayed when state is inactive
   *  @param disabledIcon icon to be displayed when state is disabled
   *         (if {@code null} the Swing standard is used)
   */
  private void init(String labelResourceKey, String tooltipResourceKey, Icon activeIcon, Icon inactiveIcon,
                    Icon disabledIcon)
  {
    setProperty(LABEL_RESOURCE_KEY_PROPERTY, labelResourceKey);
    setProperty(TOOLTIP_RESOURCE_KEY_PROPERTY, tooltipResourceKey);
    setProperty(ACTIVE_ICON_PROPERTY, activeIcon);
    setProperty(INACTIVE_ICON_PROPERTY, inactiveIcon);
    setProperty(DISABLED_ICON_PROPERTY, disabledIcon);

    I18n.addLocalizationChangeListener(new WeakReferencedLocalizable(this));

    setLocale(I18n.getDefaultLocale());
  }

  /**
   *  Create a visible state with default icons. The inactive icon is the same as the
   *  active and the disabled icon is the standard disabled view.
   *  @param labelResourceKey resource key for labels and menu items
   *  @param tooltipResourceKey resource key for tooltips
   *  @param activeIcon icon to be displayed when state is active
   */
  public VisibleState(String       labelResourceKey,
                      String       tooltipResourceKey,
                      Icon         activeIcon)
  {
    this(labelResourceKey, tooltipResourceKey, activeIcon, activeIcon,  null);
  }

  /**
   *  Create a visible state from I18n settings.
   *  @param baseTag   I18n base tag
   */
  public VisibleState(@NotNull String baseTag)
  {
    Icon normal = SwingHelper.loadIconResource(I18n.getString(baseTag + I18n.SUFFIX_ICON));
    Icon disabled = null;
    Icon inactive = normal;
    try {
      inactive = SwingHelper.loadIconResource(I18n.getString(baseTag+ I18n.SUFFIX_INACTIVE_ICON));
    } catch (MissingResourceException ignored) {
    }
    try {
      disabled = SwingHelper.loadIconResource(I18n.getString(baseTag+ I18n.SUFFIX_DISABLED_ICON));
    } catch (MissingResourceException ignored) {
    }
    init(baseTag+I18n.SUFFIX_TEXT,
         baseTag+I18n.SUFFIX_TOOLTIP,
         normal,
         inactive,
         disabled);
  }

  /**
   *  Set the locale.
   *  @param  l   locale to set.
   */
  @Override
  public void setLocale(Locale l)
  {
    locale = l;
    Object key = getValue(LABEL_RESOURCE_KEY_PROPERTY);
    if (key != null) {
      setProperty(LABEL_TEXT_PROPERTY,  I18n.getString(key.toString(), l));
    }
    key = getValue(TOOLTIP_RESOURCE_KEY_PROPERTY);
    if (ResourcedAction.I18N_DEBUG) {
      if (key == null) {
        key = getValue(LABEL_RESOURCE_KEY_PROPERTY);
        if (key != null) {
          String tag = key.toString();
          if (tag.endsWith(I18n.SUFFIX_TEXT)) {
            tag = tag.substring(0, tag.length() - I18n.SUFFIX_TEXT.length());
          }
          setProperty(TOOLTIP_TEXT_PROPERTY, String.format("[%s]", tag));
        }
      }
      else {
        String tag = key.toString();
        String ttt = I18n.getString(tag, l);
        if (tag.endsWith(I18n.SUFFIX_TOOLTIP)) {
          tag = tag.substring(0, tag.length() - I18n.SUFFIX_TOOLTIP.length());
        }
        setProperty(TOOLTIP_TEXT_PROPERTY, String.format("%s [%s]", ttt, tag));
      }
    }
    else {
      if (key != null) {
        setProperty(TOOLTIP_TEXT_PROPERTY, I18n.getString(key.toString(), l));
      }
    }
  }

  /**
   *  Get the locale.
   *  @return  the locale
   */
  @Override
  public Locale getLocale()
  {
    return locale;
  }

}
