// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import javax.swing.*;
import javax.swing.text.DefaultFormatter;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.ParseException;

/**
 * Text field for inputting an URL.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class UrlTextField
        extends JFormattedTextField
{
  private static final long serialVersionUID = 8442238956811768503L;

  /**
   * Constructor.
   * @param allowEmpty allow empty URLs represented as {@code null} values?
   */
  public UrlTextField(boolean allowEmpty)
  {
    super(new UrlFormatter(allowEmpty));
  }

  /**
   * Constructor.
   * This does not allow an empty URL.
   * @param url initial URL
   */
  public UrlTextField(@NotNull URL url)
  {
    super(new UrlFormatter(false));
    setValue(url);
  }

  /**
   * Constructor.
   * @param url         initial URL
   * @param allowEmpty  allow empty URLs represented as {@code null} values?
   */
  public UrlTextField(@Nullable URL url,
                      boolean allowEmpty)
  {
    super(new UrlFormatter(allowEmpty));

    setValue(url);
  }

  /**
   * URL formatter.
   */
  public static class UrlFormatter
          extends DefaultFormatter
  {
    private static final long serialVersionUID = -3934259494506014047L;

    /** Allow empty (i.e. {@code null}) URL? */
    private final boolean allowEmpty;

    /**
     * Constructor.
     * @param allowEmpty allow empty text representing a {@code null} URL?
     */
    public UrlFormatter(boolean allowEmpty)
    {
      this.allowEmpty = allowEmpty;
    }

    /**
     * Parses {@code text} returning an arbitrary Object. Some
     * formatters may return null.
     *
     * @param text String to convert
     * @return Object representation of text
     * @throws java.text.ParseException if there is an error in the conversion
     */
    @Override
    public Object stringToValue(String text) throws ParseException
    {
      if (text == null || text.isEmpty()) {
        if (allowEmpty) {
          return null;
        }
        throw new ParseException("URL is empty!", 0);
      }
      try {
        return new URL(text);
      } catch (MalformedURLException e) {
        throw new ParseException(text, 0);
      }
    }

    /**
     * Returns the string value to display for {@code value}.
     *
     * @param value Value to convert
     * @return String representation of value
     * @throws java.text.ParseException if there is an error in the conversion
     */
    @Override
    public String valueToString(Object value) throws ParseException
    {
      if (value == null) {
        if (allowEmpty) {
          return "";
        }
        throw new ParseException("Null value not allowed!", 0);
      }
      return value.toString();
    }

    /**
     * Clone.
     * @return clone
     * @throws CloneNotSupportedException on cloning errors
     */
    @Override
    public UrlFormatter clone() throws CloneNotSupportedException
    {
      return (UrlFormatter)super.clone();
    }
  }
}
