// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2013-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.Nullable;

import javax.swing.*;
import java.awt.*;

/**
 * Checkbox with three states: selected, unselected and partially selected.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class TriStateCheckBox
        extends JCheckBox
{
  private static final long serialVersionUID = -6279916149289195607L;

  private static class TriStateCheckBoxModel
          extends JToggleButton.ToggleButtonModel
  {
    private static final long serialVersionUID = 163885317429832053L;
    private boolean partiallySelected;

    /**
     * Set whether this checkbox is partially selected.
     * @param b partially selected state
     */
    public void setPartiallySelected(boolean b)
    {
      if (partiallySelected != b) {
        partiallySelected = b;
        super.setSelected(false);
      }
    }

    /**
     * Sets the selected state of the button.
     *
     * @param b true selects the toggle button,
     *          false deselects the toggle button.
     */
    @Override
    public void setSelected(boolean b)
    {
      partiallySelected = false;
      super.setSelected(b);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isArmed()
    {
      return partiallySelected || super.isArmed();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isPressed()
    {
      return partiallySelected || super.isPressed();
    }

    /**
     * Is this partially selected?
     * @return the answer
     */
    public boolean isPartiallySelected()
    {
      return partiallySelected;
    }
  }

  /** Icon to be displayed if partially selected. */
  private Icon partialIcon;
  /** Stored unselected icon. */
  private Icon unselectedIcon;
  /** Stored selected icon. */
  private Icon selectedIcon;

  /**
   * Creates an initially unselected check box button with no text, no icon.
   */
  public TriStateCheckBox()
  {
    this(null, null, false);
  }

  /**
   * Creates an initially unselected check box with an icon.
   *
   * @param icon the Icon image to display
   */
  public TriStateCheckBox(Icon icon)
  {
    this(null, icon, false);
  }

  /**
   * Creates a check box with an icon and specifies whether
   * or not it is initially selected.
   *
   * @param icon     the Icon image to display
   * @param selected a boolean value indicating the initial selection
   *                 state. If {@code true} the check box is selected
   */
  public TriStateCheckBox(Icon icon, boolean selected)
  {
    this(null, icon, selected);
  }

  /**
   * Creates an initially unselected check box with text.
   *
   * @param text the text of the check box.
   */
  public TriStateCheckBox(String text)
  {
    this(text, null, false);
  }

  /**
   * Creates a check box where properties are taken from the
   * Action supplied.
   * @param a action to apply (compare {@link #setAction(Action)})
   * @since 1.3
   */
  public TriStateCheckBox(Action a)
  {
    setModel(new TriStateCheckBoxModel());
    setAction(a);
  }

  /**
   * Creates a check box with text and specifies whether
   * or not it is initially selected.
   *
   * @param text     the text of the check box.
   * @param selected a boolean value indicating the initial selection
   *                 state. If {@code true} the check box is selected
   */
  public TriStateCheckBox(String text, boolean selected)
  {
    this(text, null, selected);
  }

  /**
   * Creates an initially unselected check box with
   * the specified text and icon.
   *
   * @param text the text of the check box.
   * @param icon the Icon image to display
   */
  public TriStateCheckBox(String text, Icon icon)
  {
    this(text, icon, false);
  }

  /**
   * Creates a check box with text and icon,
   * and specifies whether or not it is initially selected.
   *
   * @param text     the text of the check box.
   * @param icon     the Icon image to display
   * @param selected a boolean value indicating the initial selection
   *                 state. If {@code true} the check box is selected
   */
  public TriStateCheckBox(String text, Icon icon, boolean selected)
  {
    // Create the model
    setModel(new TriStateCheckBoxModel());

    model.setSelected(selected);

    // initialize
    init(text, icon);
  }

  /**
   * Set the tri state.
   * @param state {@code true}: checkbox is selected<br>
   *              {@code false}: checkbox is unselected<br>
   *              {@code null}: checkbox is partially selected
   */
  public void setTriState(@Nullable Boolean state)
  {
    if (state == null) {
      ((TriStateCheckBoxModel)getModel()).setPartiallySelected(true);
      if (partialIcon != null) {
        setSelectedIcon(partialIcon);
        setIcon(partialIcon);
      }
    }
    else {
      setIcon(unselectedIcon);
      setSelectedIcon(selectedIcon);
      setSelected(state);
    }
  }

  /**
   * Set the icons.
   * @param unselectedIcon  unselected icon
   * @param selectedIcon    selected icon
   * @param partialIcon     partially selected icon
   */
  public void setIcons(Icon unselectedIcon,
                       Icon selectedIcon,
                       Icon partialIcon)
  {
    this.unselectedIcon = unselectedIcon;
    this.selectedIcon = selectedIcon;
    this.partialIcon = partialIcon;
    setTriState(isPartiallySelected()
                        ? null
                        : isSelected());
  }

  /**
   * Is the box partially selected?
   * @return {@code true}; the box is partially selected<br>
   *         {@code false}: the box is completely selected or unselected
   */
  public boolean isPartiallySelected()
  {
    return ((TriStateCheckBoxModel)getModel()).isPartiallySelected();
  }

  @Override
  protected void paintComponent(Graphics g)
  {
    super.paintComponent(g);
  }
}
