// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import javax.swing.*;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Predicate;

/**
 * Some useful methods for JTree.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class TreeHelper
{
  /**
   * Get the path of a tree node.
   * @param node tree node for which the path is requested
   * @return tree path of the given node
   */
  @NotNull
  public static TreePath nodeToPath(@NotNull TreeNode node)
  {
    final LinkedList<TreeNode> nodeChain = new LinkedList<>();
    while (node != null) {
      nodeChain.addFirst(node);
      node = node.getParent();
    }
    return new TreePath(nodeChain.toArray());
  }

  /**
   * Traverse the tree and get the first following node which fulfills a check.
   * This method performs a depth-first traversal.
   * @param model       tree model
   * @param currentNode current node, if {@code null} the search starts with the root node,
   *                    if not the first possible find is the node following this node
   * @param checker     checker which tells whether a node fulfills the search criteria
   *                    (e.g. if you are only interested in leafs, just check for that)
   * @return the first node found which fulfills the search criteria, {@code null}
   *         if no following node fulfills the criteria
   * @see #getNextMatchCycling(TreeModel, TreeNode, Predicate)
   */
  @Nullable
  public static TreeNode getNextMatch(@NotNull TreeModel model,
                                      @Nullable TreeNode currentNode,
                                      @NotNull Predicate<TreeNode> checker)
  {
    TreeNode node = currentNode == null
            ? (TreeNode)model.getRoot()
            : nextNode(currentNode);
    while (node != null) {
      if (checker.test(node)) {
        return node;
      }
      node = nextNode(node);
    }
    // not found
    return null;
  }

  /**
   * Traverse the tree and get the first node which fulfills a check.
   * This method performs a depth-first traversal.
   * This will cycle through nodes before the given {@code currentNode} if nothing
   * is found following it.
   * @param model       tree model
   * @param currentNode current node, if {@code null} the search starts with the root node,
   *                    if not the first possible find is the node following this node
   * @param checker     checker which tells whether a node fulfills the search criteria
   *                    (e.g. if you are only interested in leafs, just check for that)
   * @return the first node found which fulfills the search criteria, maybe {@code currentNode}
   *         if it's the only node fulfilling it, or {@code null} if no node (including
   *         {@code currentNode}) fulfills the criteria
   * @see #getNextMatch(TreeModel, TreeNode, Predicate)
   */
  @Nullable
  public static TreeNode getNextMatchCycling(@NotNull TreeModel model,
                                             @Nullable TreeNode currentNode,
                                             @NotNull Predicate<TreeNode> checker)
  {
    final TreeNode result = getNextMatch(model, currentNode, checker);
    if (currentNode != null  &&  result == null) {
      // cycle
      return getNextMatch(model, null, checker);
    }
    return result;
  }

  /**
   * Get the next node when traversing the tree depth first.
   * @param node current node
   * @return the node following the given node
   */
  @Nullable
  public static TreeNode nextNode(@NotNull TreeNode node)
  {
    return nextNode(node, true);
  }

  /**
   * Get the next node when traversing the tree depth first.
   * @param node current node
   * @param withChildren include child nodes in search?
   * @return the node following the given node
   */
  @Nullable
  private static TreeNode nextNode(@NotNull TreeNode node, boolean withChildren)
  {
    if (withChildren && node.getAllowsChildren() && node.getChildCount() > 0) {
      return node.getChildAt(0);
    }
    final TreeNode parent = node.getParent();
    if (parent == null) {
      return null;
    }
    final int childCount = parent.getChildCount();
    final int index = parent.getIndex(node);
    if (index < childCount - 1) {
      return parent.getChildAt(index + 1);
    }
    return nextNode(parent, false);
  }

  /**
   * Traverse the tree and get all nodes fulfilling a check.
   * @param model    tree model
   * @param checker  node checker
   * @return all nodes for which the node checker considered a match
   */
  @NotNull
  public static Collection<TreeNode> getMatches(@NotNull TreeModel model,
                                                @NotNull Predicate<TreeNode> checker)
  {
    TreeNode node = (TreeNode)model.getRoot();
    Collection<TreeNode> result = new LinkedList<TreeNode>();
    while (node != null) {
      if (checker.test(node)) {
        result.add(node);
      }
      node = nextNode(node);
    }
    return result;
  }

  /**
   * Traverse over the tree and call a consumer for each node.
   * @param model    tree model to traverse over
   * @param consumer consumer to be called for node
   */
  public static void traverse(@NotNull TreeModel model,
                              @NotNull Consumer<TreePath> consumer)
  {
    final Object root = model.getRoot();
    if (root != null) {
      traverse(new TreePath(root), consumer);
    }
  }

  /**
   * Traverse over a (sub) tree and call a consumer for each node.
   * @param path     tree path to traverse
   * @param consumer consumer called for each node
   */
  public static void traverse(@NotNull TreePath path,
                              @NotNull Consumer<TreePath> consumer)
  {
    consumer.accept(path);
    final TreeNode node = (TreeNode)path.getLastPathComponent();
    final Enumeration<?> e = node.children();
    if (e != null) {
      while (e.hasMoreElements()) {
        final TreeNode child = (TreeNode)e.nextElement();
        traverse(path.pathByAddingChild(child), consumer);
      }
    }
  }

  /**
   * Is a node currently visible for the user?
   * @param tree tree for which the node is defined
   * @param path path to node
   * @return {@code true} if the node is expanded and displayed in the viewport<br>
   *         {@code false} otherwise
   */
  public static boolean isDisplayed(@NotNull final JTree tree,
                                    final TreePath path)
  {
    return tree.isVisible(path) &&
           tree.getVisibleRect().intersects(tree.getRowBounds(tree.getRowForPath(path)));
  }

  /**
   * Does a tree path start with a given prefix?
   * I.e. is the tree node defined by {@code path} a (grand*)child of {@code prefix}
   * (or the same)?
   * @param path   path to check
   * @param prefix prefix path to check
   * @return {@code true} if {@code path} itself or one of its successive parents equals {@code prefix},<br>
   *         {@code false} otherwise
   */
  public static boolean pathStartsWith(@NotNull TreePath path,
                                       @NotNull TreePath prefix)
  {
    if (path.getPathCount() < prefix.getPathCount()) {
      return false;
    }
    final Object[] prefixArray = path.getPath();
    final Object[] pathArray = Arrays.copyOf(path.getPath(), prefixArray.length);
    return Arrays.equals(prefixArray, pathArray);
  }
}
