// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 *  Handle a bunch of states and take care that only one is active.
 *
 *  @author  <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class StateManager
  implements PropertyChangeListener
{
  /** A collection of states. */
  private final List<State> stateList = new LinkedList<>();
  /** The active state. */
  private State activeState;
  /** Do we have toggable states? */
  private boolean haveToggleStates = false;
  /** Is this manager temporarily ignoring changes? */
  private boolean ignoringChanges;

  /**
   *  Add a state.
   *  @param  state  state to add
   */
  public void addState(@NotNull State state)
  {
    state.setManager(this);
    stateList.add(state);

    state.addPropertyChangeListener(this);

    if (!state.isToggleEnabled() && activeState == null) {
      // one state has to be active, if it is not toggled
      state.activate();
      activeState = state;
    }
    else if (state.isActivated()) {
      // if the state is added as active, use it instead
      setActiveState(state);
    }
    haveToggleStates |= state.isToggleEnabled();
  }

  /**
   *  Set the currently active state.
   *  @param state  state to set active
   */
  void setActiveState(@Nullable State state)
  {
    if (activeState != state) {
      if (stateList.contains(state)  ||
          (state == null && haveToggleStates)) {
        if (activeState != null) {
          activeState.deactivate();
        }
        activeState = state;
      }
      else {
        throw new RuntimeException("Trying to activate unregistered state");
      }
    }
  }

  /**
   *  Get the list of known states.
   *  @return list of states
   */
  @NotNull
  public List<State> getStates()
  {
    return Collections.unmodifiableList(stateList);
  }

  /**
   *  Get the active state.
   *  @return the active state (if states may be toggled this may be {@code null})
   */
  @Nullable
  public State getActiveState()
  {
    return activeState;
  }

  /**
   *  May a state be activated?
   *  @param state state to check
   *  @return {@code true} if the state is known and enabled
   */
  boolean mayActivate(@NotNull AbstractState state)
  {
    if (stateList.contains(state)) {
      return state.isEnabled();
    }
    else {
      throw new RuntimeException("Trying to access unregistered state");
    }
  }

  /**
   *  May the state be enabled or disabled?
   *  @param state the state
   *  @param enable enable state
   *  @return {@code true} if enable is true or this is not the only enabled state
   */
  boolean mayEnable(AbstractState state, boolean enable)
  {
    return enable || getFirstEnabledState(state) != null;
  }

  /**
   *  Get the first state which is enabled.
   *  @param ignore state to be ignored during search
   *  @return first state which is enabled
   */
  protected State getFirstEnabledState(AbstractState ignore)
  {
    for (State state : stateList) {
      if (state != ignore && state.isEnabled()) {
        return state;
      }
    }

    // nothing found
    return null;
  }

  /**
   * This method gets called when a bound property is changed.
   * @param evt A PropertyChangeEvent object describing the event source
   *   	and the property that has changed.
   */
  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    if (ignoringChanges) {
      return;
    }
    AbstractState state = (AbstractState)evt.getSource();
    if (AbstractState.ACTIVATION_PROPERTY.equals(evt.getPropertyName())) {
      if (((Boolean)evt.getNewValue()).booleanValue()) {
        setActiveState(state);
      }
      else if (haveToggleStates  &&  state == activeState) {
        setActiveState(null);
      }
    }
    else if (AbstractState.ENABLE_PROPERTY.equals(evt.getPropertyName())) {
      if (!((Boolean)evt.getNewValue()).booleanValue()) {
        if (activeState == state  &&  !haveToggleStates) {
          // set first enabled state active
          getFirstEnabledState(state).activate();
        }
      }
    }
  }

  /**
   * Is this state manager ignoring changes?
   * @return ignoring changes
   * @see #setIgnoringChanges(boolean)
   */
  public boolean isIgnoringChanges()
  {
    return ignoringChanges;
  }

  /**
   * Set whether this state manager is ignoring changes.
   * This should only be enabled temporarily when complex actions may cause conflicts by the automatisms of
   * this manager. Making calling this method necessary is in most cases the result of a design flaw.
   * @param ignoringChanges {@code true} the state manager is not propagating state changes<br>
   *                        {@code false} the state manager propagates state changes (normal mode)
   */
  public void setIgnoringChanges(boolean ignoringChanges)
  {
    this.ignoringChanges = ignoringChanges;
  }
}
