// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.util.Utility;
import de.caff.vic.RecolorableIcon;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.GeneralPath;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import static de.caff.vic.RecoloringVisitor.TO_LIGHTER_GRAY;

/**
 *  Button which displays a {@link State}.
 *
 *  @author  <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class StateButton
  extends    JToggleButton
  implements PropertyChangeListener,
             ItemListener
{
  private static final long serialVersionUID = 6719512758299692925L;
  /** The state which this button displays. */
  private final State state;

  /**
   *  Create a new button from a state.
   *  @param state state from which to create the button
   */
  public StateButton(final State state)
  {
    setModel(new ToggleButtonModel() {
      private static final long serialVersionUID = 3697751640306107911L;

      /**
       * Sets the button to pressed or unpressed.
       *
       * @param b true to set the button to "pressed"
       */
      @Override
      public void setPressed(boolean b) {
        if (!state.isActivated()  ||  b) {
          super.setPressed(b);
        }
        else if (state.isToggleEnabled()) {
          super.setPressed(b);
        }
      }
    });

    this.state = state;

    Icon icon = (Icon)state.getValue(State.INACTIVE_ICON_PROPERTY);
    if (icon != null) {
      setIcon(icon);

      Icon activeIcon = (Icon)state.getValue(State.ACTIVE_ICON_PROPERTY);
      if (activeIcon != null) {
        setSelectedIcon(activeIcon);
      }

      Icon disabledIcon = (Icon)state.getValue(State.DISABLED_ICON_PROPERTY);
      if (disabledIcon != null) {
        setDisabledIcon(disabledIcon);
        setDisabledSelectedIcon(disabledIcon);
      }
      else {
        if (icon instanceof RecolorableIcon) {
          final RecolorableIcon recoloredIcon =
                  ((RecolorableIcon)icon).getRecoloredIcon(TO_LIGHTER_GRAY);
          setDisabledIcon(recoloredIcon);
          if (activeIcon == null) {
            setDisabledSelectedIcon(recoloredIcon);
          }
        }
        if (activeIcon instanceof RecolorableIcon) {
          setDisabledSelectedIcon(((RecolorableIcon)activeIcon).getRecoloredIcon(TO_LIGHTER_GRAY));
        }
      }
    }
    else {
      setText((String)state.getValue(State.LABEL_TEXT_PROPERTY));
    }

    String ttt = (String)state.getValue(State.TOOLTIP_TEXT_PROPERTY);
    if (ttt != null) {
      setToolTipText(ttt);
    }

    setEnabled(state.isEnabled());
    setSelected(state.isActivated());

    addItemListener(this);
    state.addPropertyChangeListener(this);

    addMouseListener(new MouseAdapter() {
      /**
       * Invoked when a mouse button has been pressed on a component.
       */
      @Override
      public void mousePressed(MouseEvent e)
      {
        maybeShowPopup(e);
      }

      /**
       * Invoked when a mouse button has been released on a component.
       */
      @Override
      public void mouseReleased(MouseEvent e)
      {
        maybeShowPopup(e);
      }

      /**
       *  Show a popup if it's the trigger on this os.
       *  @param e mouse event
       */
      private void maybeShowPopup(MouseEvent e) {
        if (state.isActivated()  &&  e.isPopupTrigger()) {
          state.showPopup(e.getComponent(),
                          e.getX(), e.getY());
        }
      }
    });
  }

  /**
   * This method gets called when a bound property is changed.
   * @param evt A PropertyChangeEvent object describing the event source
   *   	and the property that has changed.
   */
  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    String property = evt.getPropertyName();

    if (State.ACTIVATION_PROPERTY.equals(property)) {
      boolean myState = (Boolean)evt.getNewValue();
      if (myState != isSelected()) {
        setSelected(myState);
      }
    }
    else if (State.ENABLE_PROPERTY.equals(property)) {
      boolean myState = (Boolean)evt.getNewValue();
      if (myState != isEnabled()) {
        setEnabled(myState);
      }
    }
    else if (State.ACTIVE_ICON_PROPERTY.equals(property)) {
      Icon icon = (Icon)evt.getNewValue();
      if (!icon.equals(getSelectedIcon())) {
        setSelectedIcon(icon);
      }
    }
    else if (State.DISABLED_ICON_PROPERTY.equals(property)) {
      Icon icon = (Icon)evt.getNewValue();
      if (!icon.equals(getDisabledIcon())) {
        setDisabledIcon(icon);
        setDisabledSelectedIcon(icon);
      }
    }
    else if (State.INACTIVE_ICON_PROPERTY.equals(property)) {
      Icon icon = (Icon)evt.getNewValue();
      if (!icon.equals(getIcon())) {
        setIcon(icon);
      }
    }
    else if (State.LABEL_TEXT_PROPERTY.equals(property)) {
      String text = (String)evt.getNewValue();
      if (getIcon() == null) {
        setText(text);
      }
    }
    else if (State.TOOLTIP_TEXT_PROPERTY.equals(property)) {
      String text = (String)evt.getNewValue();
      setToolTipText(text);
    }
    else if (State.POPUP_PROPERTY.equals(property)) {
      repaint();
    }
  }

  /**
   * Invoked when an item has been selected or deselected by the user.
   * The code written for this method performs the operations
   * that need to occur when an item is selected (or deselected).
   */
  @Override
  public void itemStateChanged(ItemEvent e)
  {
    if (e.getStateChange() == ItemEvent.SELECTED) {
      state.activate();
    }
    else {
      state.deactivate();
    }
  }

  /**
   * Calls the UI delegate's paint method, if the UI delegate
   * is non-{@code null}.  We pass the delegate a copy of the
   * {@code Graphics} object to protect the rest of the
   * paint code from irrevocable changes
   * (for example, {@code Graphics.translate}).
   * <p>
   * If you override this in a subclass you should not make permanent
   * changes to the passed in {@code Graphics}. For example, you
   * should not alter the clip {@code Rectangle} or modify the
   * transform. If you need to do these operations you may find it
   * easier to create a new {@code Graphics} from the passed in
   * {@code Graphics} and manipulate it. Further, if you do not
   * invoker super's implementation you must honor the opaque property,
   * that is
   * if this component is opaque, you must completely fill in the background
   * in a non-opaque color. If you do not honor the opaque property you
   * will likely see visual artifacts.
   * <p>
   * The passed in {@code Graphics} object might
   * have a transform other than the identify transform
   * installed on it.  In this case, you might get
   * unexpected results if you cumulatively apply
   * another transform.
   *
   * @param g the {@code Graphics} object to protect
   * @see #paint
   * @see javax.swing.plaf.ComponentUI
   */
  @Override
  protected void paintComponent(Graphics g)
  {
    super.paintComponent(g);
    if (isEnabled() && isSelected() && state.hasPopup()) {
      // show indicator
      Dimension size = getSize();
      Graphics2D g2 = (Graphics2D)g;
      paintPopupIndicator(g2, getBackground(), size);
    }
  }

  /**
   * Paint an indicator signaling that this state provides a popup menu.
   * @param g2        graphics to paint to
   * @param bgColor   background color of component
   * @param size      size of component
   */
  public static void paintPopupIndicator(Graphics2D g2, Color bgColor, Dimension size)
  {
    if (size.width < 8  ||  size.height < 8) {
      return;
    }
    Color color = Utility.getPhysiologicalBrightness(bgColor) >= 0.5f ?
            Color.black :
            Color.white;
    g2.setColor(new Color(color.getRed(),
                          color.getGreen(),
                          color.getBlue(),
                          128));
    GeneralPath triangle = new GeneralPath();
    final int border = 3;
    float len = 0.25f*Math.min(size.width, size.height) - 1;
    triangle.moveTo(size.width - border, size.height - border);
    triangle.lineTo(size.width - border, size.height - border - len);
    triangle.lineTo(size.width - border - len, size.height - border);
    triangle.closePath();
    g2.fill(triangle);
  }
}
