// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;

import javax.swing.*;
import java.awt.*;
import java.beans.PropertyChangeListener;

/**
 *  State is an interface for states, especially in GUIs. It is similar
 *  to the Action interface of Swing.
 *
 *  @author  <a href="mailto:rammi@caff.de">Rammi</a>
 */
public interface State
        extends Action,
                ToolBarElement
{
  /** Property identifier for enable state. */
  String ENABLE_PROPERTY        = "ENABLE";
  /** Property identifier for activation. */
  String ACTIVATION_PROPERTY    = "ACTIVATION";
  /** Property name for active icon. */
  String ACTIVE_ICON_PROPERTY   = "ACTIVE_ICON";
  /** Property name for inactive icon. */
  String INACTIVE_ICON_PROPERTY = "INACTIVE_ICON";
  /** Property name for disabled icon. */
  String DISABLED_ICON_PROPERTY = "DISABLED_ICON";
  /** Property name for label text. */
  String LABEL_TEXT_PROPERTY    = "LABEL_TEXT";
  /** Property name for tooltip text property. */
  String TOOLTIP_TEXT_PROPERTY  = "TOOLTIP_TEXT";

  /** Property name for the possibility to show a popup menu. */
  String POPUP_PROPERTY = "de.caff.gimmicks.swing.State.POPUP";

  /**
   *  Set the state manager which handles this state.
   *  @param stateManager state manager
   */
  void setManager(@NotNull StateManager stateManager);

  /**
   *  Add a property change listener for a special property.
   *  @param property property identifier
   *  @param listener listener to add
   */
  void addPropertyChangeListener(@NotNull String property, @NotNull PropertyChangeListener listener);

  /**
   *  Remove a property change listener for a special property.
   *  @param property property identifier
   *  @param listener listener to remove
   */
  void removePropertyChangeListener(@NotNull String property, @NotNull PropertyChangeListener listener);

  /**
   *  Activate this state.
   *  @return {@code true} if the activation was possible,
   *          {@code false} otherwise
   */
  boolean activate();

  /**
   *  Is this state activated?
   *  @return the answer
   */
  boolean isActivated();

  /**
   *  Deactivate this state.
   */
  void deactivate();

  /**
   * Is it possible to switch this state off by clicking on it with the mouse?
   * @return the answer
   */
  boolean isToggleEnabled();

  /**
   *  Show a popup menu.
   *  Implementations are free to do nothing.
   *  @param comp component where to show the popup menu
   *  @param x    x position in component
   *  @param y    y position in component
   *  @see #hasPopup()
   */
  void showPopup(@NotNull Component comp, int x, int y);

  /**
   * Does this state provide a popup menu?
   * If the answer is {@code true} {@link #showPopup(java.awt.Component, int, int)}
   * is expected to display a popup menu.
   * @return the answer
   * @see #POPUP_PROPERTY
   */
  boolean hasPopup();

  /**
   * Refresh state.
   */
  void refresh();
}
