// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.i18n.I18n;

import javax.swing.filechooser.FileFilter;
import java.io.File;

/**
 *  A simple file filter for the Swing file chooser allowing only files with a certain extension.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SimpleFileExtensionFilter
        extends FileFilter
{
  static {
    I18n.addAppResourceBase("de.caff.gimmicks.swing.GimmicksSwingResourceBundle");
  }

  /** The description of this filter. */
  @NotNull
  private final String description;
  /** The extension to look for. */
  @NotNull
  private final String extension;
  /** Allow directories? */
  private boolean allowDirectories = true;

  /**
   *  Create a file filter with a default name.
   *  @param extension the extension to look for
   */
  public SimpleFileExtensionFilter(@NotNull String extension)
  {
    if (!extension.startsWith(".")) {
      extension = "."+extension;
    }
    this.extension = extension.toUpperCase();
    description = I18n.format("nmFileFilter", this.extension.substring(1));
  }

  /**
   *  Create a file filter.
   *  @param extension the extension to look for
   *  @param description description of this filter
   */
  public SimpleFileExtensionFilter(@NotNull String extension, @NotNull String description)
  {
    if (!extension.startsWith(".")) {
      extension = "."+extension;
    }
    this.extension = extension.toUpperCase();
    this.description = description;
  }

  /**
   * The description of this filter. For example: "JPEG Images"
   *
   * @see javax.swing.filechooser.FileView#getName
   */
  @NotNull
  @Override
  public String getDescription()
  {
    return description;
  }

  /**
   * Whether the given file is accepted by this filter.
   */
  @Override
  public boolean accept(File f)
  {
    if (f.isDirectory()) {
      return allowDirectories;
    }
    return f.getName().toUpperCase().endsWith(extension);
  }

  /**
   *  Are directories allowed by this filter?
   *  @return the answer
   */
  public boolean isAllowDirectories()
  {
    return allowDirectories;
  }

  /**
   *  Set whether directories are allowed by this filter.
   *  @param allowDirectories allow directories?
   */
  public void setAllowDirectories(boolean allowDirectories)
  {
    this.allowDirectories = allowDirectories;
  }

  /**
   *  Add the extension to a path if it does not already end with it and the file is non-existent.
   *  @param file fle which paths is needed
   *  @return path including the extension
   */
  @NotNull
  public String getFixedPath(@NotNull File file)
  {
    String path = file.getPath();
    if (!file.exists()  &&  !path.toUpperCase().endsWith(extension)) {
      path += extension.toLowerCase();
    }
    return path;
  }

  /**
   *  Add the extension to a path if it does not already end with it and the file is non-existent.
   *  @param file fle which paths is needed
   *  @return path including the extension
   */
  @NotNull
  public File getFixedFilePath(@NotNull File file)
  {
    final String path = file.getPath();
    if (!file.exists()  &&  !path.toUpperCase().endsWith(extension)) {
      return new File(path + extension.toLowerCase());
    }
    return file;
  }
}
