// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;

import javax.swing.*;
import javax.swing.plaf.basic.BasicHTML;
import javax.swing.plaf.basic.BasicLabelUI;
import javax.swing.text.View;
import java.awt.*;

/**
 * UI for painting labels rotated at 90°.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since April.28, 2016
 */
public class RotatedLabelUI
        extends BasicLabelUI
{
  private static final double ROTATION_ANGLE = 0.5 * Math.PI;
  private final boolean counterClockWise;
  @NotNull
  private final Rectangle paintIconR = new Rectangle();
  @NotNull
  private final Rectangle paintTextR = new Rectangle();

  private static final int HORIZONTAL_MARGIN = 0;
  private static final int VERTICAL_MARGIN = 3;


  /**
   * Constructor.
   * @param counterClockWise {@code true}: rotate 90° CCW,<br>
   *                         {@code false}: rotate 90° CW
   */
  public RotatedLabelUI(boolean counterClockWise)
  {
    this.counterClockWise = counterClockWise;
  }

  @Override
  public int getBaseline(JComponent c, int width, int height)
  {
    return super.getBaseline(c, width, height);
  }

  @Override
  public Dimension getPreferredSize(JComponent c)
  {
    final Dimension unrotated = super.getPreferredSize(c);
    return new Dimension(unrotated.height + 2*HORIZONTAL_MARGIN,
                         unrotated.width + 2*VERTICAL_MARGIN);
  }

  @Override
  public Dimension getMinimumSize(JComponent c)
  {
    final Dimension unrotated = super.getMinimumSize(c);
    return new Dimension(unrotated.height + 2*HORIZONTAL_MARGIN,
                         unrotated.width + 2*VERTICAL_MARGIN);
  }

  @Override
  public Dimension getMaximumSize(JComponent c)
  {
    final Dimension unrotated = super.getMaximumSize(c);
    return new Dimension(unrotated.height + 2*HORIZONTAL_MARGIN,
                         unrotated.width + 2*VERTICAL_MARGIN);
  }


  @Override
  public void paint(Graphics g, JComponent c)
  {
    JLabel label = (JLabel)c;
    String text = label.getText();
    Icon icon = (label.isEnabled()) ? label.getIcon() : label.getDisabledIcon();

    if ((icon == null) && (text == null)) {
      return;
    }

    FontMetrics fm = label.getFontMetrics(label.getFont());
    String clippedText = layout(label, fm, c.getHeight(), c.getWidth());

    final Graphics2D g2 = ((Graphics2D)g.create());
    final int cw = c.getWidth() / 2;
    final int ch = c.getHeight() / 2;

    g2.translate(cw, ch);
    if (counterClockWise) {
      g2.rotate(-ROTATION_ANGLE);
    }
    else {
      g2.rotate(ROTATION_ANGLE);
    }
    g2.translate(-ch + VERTICAL_MARGIN, -cw);

    if (icon != null) {
      icon.paintIcon(c, g2, paintIconR.x, paintIconR.y);
    }

    if (text != null) {
      View v = (View) c.getClientProperty(BasicHTML.propertyKey);
      if (v != null) {
        v.paint(g, paintTextR);
      } else {
        int textX = paintTextR.x;
        int textY = paintTextR.y + fm.getAscent();

        if (label.isEnabled()) {
          paintEnabledText(label, g2, clippedText, textX, textY);
        }
        else {
          paintDisabledText(label, g2, clippedText, textX, textY);
        }
      }
    }
  }

  private String layout(JLabel label, FontMetrics fm,
                        int width, int height) {
    Insets insets = label.getInsets(null);
    String text = label.getText();
    Icon icon = (label.isEnabled()) ? label.getIcon() :
            label.getDisabledIcon();
    Rectangle paintViewR = new Rectangle();
    paintViewR.x = insets.left;
    paintViewR.y = insets.top;
    paintViewR.width = width - (insets.left + insets.right);
    paintViewR.height = height - (insets.top + insets.bottom);
    paintIconR.x = paintIconR.y = paintIconR.width = paintIconR.height = 0;
    paintTextR.x = paintTextR.y = paintTextR.width = paintTextR.height = 0;
    return layoutCL(label, fm, text, icon, paintViewR, paintIconR,
                    paintTextR);
  }


}
