// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2003-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;
import de.caff.i18n.WeakReferencedLocalizable;
import de.caff.util.swing.SwingHelper;
import de.caff.util.Utility;

import javax.swing.*;
import java.util.Locale;
import java.util.MissingResourceException;

/**
 *  An action which knows about resources.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class ResourcedAction
        extends AbstractAction
        implements Localizable
{
  private static final long serialVersionUID = -8362607471453797262L;
  /** Show base names on action debug? */
  static final boolean I18N_DEBUG = Utility.getBooleanParameter("i18n.action.debug", false);
  /** Use this as a prefix for resources of standard names, so they are updated automatically on locale changes. */
  protected static final String RESOURCE_PREFIX = "i18n.";
  /** Internally used length of RESOURCE_PREFIX. */
  private   static final int    RESOURCE_PREFIX_LENGTH = RESOURCE_PREFIX.length();
  /** Use this as a prefix for resources of icon pathes, so they are updated automatically on locale changes. */
  protected static final String ICON_PREFIX     = RESOURCE_PREFIX+"icon.";
  /** Internally used length of ICON_PREFIX. */
  private   static final int    ICON_PREFIX_LENGTH = ICON_PREFIX.length();
  /** The key for accelerator resources. Compare {@link #ACCELERATOR_KEY} */
  public static final String ACCELERATOR_KEY_RESOURCE    = RESOURCE_PREFIX+ACCELERATOR_KEY;
  /** The key for action command resources. Compare {@link #ACTION_COMMAND_KEY} */
  public static final String ACTION_COMMAND_KEY_RESOURCE = RESOURCE_PREFIX+ACTION_COMMAND_KEY;
  /** The key for long description resources. Compare {@link #LONG_DESCRIPTION} */
  public static final String LONG_DESCRIPTION_RESOURCE   = RESOURCE_PREFIX+LONG_DESCRIPTION;
  /** The key for mnemonic resources. Compare {@link #MNEMONIC_KEY} */
  public static final String MNEMONIC_KEY_RESOURCE       = RESOURCE_PREFIX+MNEMONIC_KEY;
  /** The key for name resources. Compare {@link #NAME} */
  public static final String NAME_RESOURCE               = RESOURCE_PREFIX+NAME;
  /** The key for short description resources. Compare {@link #SHORT_DESCRIPTION} */
  public static final String SHORT_DESCRIPTION_RESOURCE  = RESOURCE_PREFIX+SHORT_DESCRIPTION;
  /** The key for small icon path resources. Compare {@link #SMALL_ICON} */
  public static final String SMALL_ICON_RESOURCE         = ICON_PREFIX+SMALL_ICON;
  /** Boolean property for providing a popup menu, see {@link #hasPopup()}. */
  public static final String POPUP_PROPERTY = "de.caff.gimmicks.swing.ResourcedAction.POPUP";

  /** Basic I18n tag. */
  @NotNull
  private final String baseTag;
  /** The current locale. */
  private Locale locale;

  /**
   *  Create a resourced action with the default locale.
   *
   *  @param baseTag basic i18n tag.The settings of this action are tried to set by evaluating this
   *                 resource tag with one of the following suffixes:
   *                 <ul>
   *                 <li>{@link I18n#SUFFIX_TEXT}</li>
   *                 <li>{@link I18n#SUFFIX_TOOLTIP}</li>
   *                 <li>{@link I18n#SUFFIX_DESCRIPTION}</li>
   *                 <li>{@link I18n#SUFFIX_ACCELERATOR}</li>
   *                 <li>{@link I18n#SUFFIX_MNEMONIC}</li>
   *                 </ul>
   */
  protected ResourcedAction(@NotNull String baseTag)
  {
    this(baseTag, null);
  }

  /**
   *  Create a resourced action.
   *
   *  @param baseTag basic i18n tag.The settings of this action are tried to set by evaluating this
   *                 resource tag with one of the following suffixes:
   *                 <ul>
   *                 <li>{@link I18n#SUFFIX_TEXT}</li>
   *                 <li>{@link I18n#SUFFIX_TOOLTIP}</li>
   *                 <li>{@link I18n#SUFFIX_DESCRIPTION}</li>
   *                 <li>{@link I18n#SUFFIX_ACCELERATOR}</li>
   *                 <li>{@link I18n#SUFFIX_MNEMONIC}</li>
   *                 </ul>
   *  @param l       locale to use
   */
  protected ResourcedAction(@NotNull String baseTag, @Nullable Locale l)
  {
    this.baseTag = baseTag;
    setLocale(l);
    try {
      putValue(NAME_RESOURCE, baseTag+I18n.SUFFIX_TEXT);
    } catch (MissingResourceException e) {
    }
    try {
      putValue(SHORT_DESCRIPTION_RESOURCE, baseTag+I18n.SUFFIX_TOOLTIP);
    } catch (MissingResourceException e) {
    }
    try {
      putValue(LONG_DESCRIPTION_RESOURCE, baseTag+I18n.SUFFIX_DESCRIPTION);
    } catch (MissingResourceException e) {
    }
    try {
      putValue(ACCELERATOR_KEY_RESOURCE, baseTag+I18n.SUFFIX_ACCELERATOR);
    } catch (MissingResourceException e) {
    }
    try {
      putValue(MNEMONIC_KEY_RESOURCE, baseTag+I18n.SUFFIX_MNEMONIC);
    } catch (MissingResourceException e) {
    }
    try {
      putValue(SMALL_ICON_RESOURCE, baseTag+I18n.SUFFIX_ICON);
    } catch (MissingResourceException e) {
    }
    I18n.addLocalizationChangeListener(new WeakReferencedLocalizable(this));
  }

  /**
   * Get the basic I18n tag.
   * @return base tag
   */
  @NotNull
  public String getBaseTag()
  {
    return baseTag;
  }

  /**
   *  Set the locale. This updates all internal resources which depend on locales.
   *  @param l  new locale
   */
  @Override
  public void setLocale(Locale l)
  {
    if (l == null) {
      l = Locale.getDefault();
    }
    if (!l.equals(locale)) {
      locale = l;
      Object[] keys = getKeys();
      if (keys != null) {
        for (Object k : keys) {
          String key = k.toString();
          if (key.startsWith(RESOURCE_PREFIX)) {
            putValue(key, getValue(key));  // this reevaluates the locale
          }
        }
      }
    }
  }

  /**
   *  Get the current locale of this action.
   *  @return current locale
   */
  @Override
  public Locale getLocale()
  {
    return locale;
  }

  /**
   * Sets the {@code Value} associated with the specified key.
   * This takes care of setting depending keys for localized keys.
   *
   * @param key  the {@code String} that identifies the stored object
   * @param newValue the {@code Object} to store using this key
   * @see Action#putValue
   */
  @Override
  public void putValue(@NotNull String key, Object newValue)
  {
    if (key.startsWith(RESOURCE_PREFIX)) {
      final String tag = newValue.toString();
      String resolved = I18n.getString(tag, locale);

      if (I18N_DEBUG &&  tag.endsWith(I18n.SUFFIX_TOOLTIP)) {
        resolved = String.format("%s [%s]", resolved, tag.substring(0, tag.length() - I18n.SUFFIX_TOOLTIP.length()));
      }
      if (key.startsWith(ICON_PREFIX)) {
        final Icon icon = SwingHelper.loadIconResource(resolved);
        super.putValue(key.substring(ICON_PREFIX_LENGTH),
                       icon);
      }
      else if (MNEMONIC_KEY_RESOURCE.equals(key)) {
        super.putValue(MNEMONIC_KEY, (int)resolved.charAt(0));
      }
      else if (ACCELERATOR_KEY_RESOURCE.equals(key)) {
        super.putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(resolved));
      }
      else {
        super.putValue(key.substring(RESOURCE_PREFIX_LENGTH),
                       resolved);
      }
    }
    super.putValue(key, newValue);
  }

  /**
   * Clones the abstract action. This gives the clone
   * its own copy of the key/value list,
   * which is not handled for you by {@code Object.clone()}.
   */
  @Override
  protected Object clone() throws CloneNotSupportedException
  {
    return super.clone();
  }

  /**
   * Does this state provide a popup menu?
   *
   * @return the answer
   * @see #POPUP_PROPERTY
   */
  public boolean hasPopup()
  {
    return false;
  }
}
