// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2013-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.Nullable;
import de.caff.util.settings.DirectoryPreferenceProperty;
import de.caff.util.settings.SizePreferenceProperty;

import javax.swing.*;
import javax.swing.filechooser.FileSystemView;
import java.awt.*;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.io.File;

/**
 * File chooser which keeps the size between calls.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ResizableFileChooser
         extends JFileChooser
{
  private static final long serialVersionUID = 8251916931286171414L;
  /**
   * Load/store this preference property to keep size changes between separate runs of the program.
   */
  public static final SizePreferenceProperty PP_FILE_CHOOSER_SIZE = new SizePreferenceProperty("FILE_CHOOSER_SIZE");
  /** Component adapter listening for resize events. */
  private static final ComponentListener SIZE_UPDATER = new ComponentAdapter()
  {
    @Override
    public void componentResized(ComponentEvent e)
    {
      PP_FILE_CHOOSER_SIZE.setSize(e.getComponent());
    }
  };
  private final DirectoryPreferenceProperty directoryPreferenceProperty;

  /**
   * Constructs a {@code JFileChooser} pointing to the user's
   * default directory. This default depends on the operating system.
   * It is typically the "My Documents" folder on Windows, and the
   * user's home directory on Unix.
   */
  public ResizableFileChooser()
  {
    this.directoryPreferenceProperty = null;
  }

  /**
   * Constructs a {@code JFileChooser} using the given path.
   * Passing in a {@code null}
   * string causes the file chooser to point to the user's default directory.
   * This default depends on the operating system. It is
   * typically the "My Documents" folder on Windows, and the user's
   * home directory on Unix.
   *
   * @param currentDirectoryPath a {@code String} giving the path
   *                             to a file or directory
   */
  public ResizableFileChooser(@Nullable String currentDirectoryPath)
  {
    super(currentDirectoryPath);
    this.directoryPreferenceProperty = null;
  }

  /**
   * Constructs a {@code JFileChooser} using the given {@code File}
   * as the path. Passing in a {@code null} file
   * causes the file chooser to point to the user's default directory.
   * This default depends on the operating system. It is
   * typically the "My Documents" folder on Windows, and the user's
   * home directory on Unix.
   *
   * @param currentDirectory a {@code File} object specifying
   *                         the path to a file or directory
   */
  public ResizableFileChooser(@Nullable File currentDirectory)
  {
    super(currentDirectory);
    this.directoryPreferenceProperty = null;
  }

  /**
   * Constructs a {@code JFileChooser} using the given
   * {@code FileSystemView}.
   * @param fsv                         file system view, or {@code null} for default
   */
  public ResizableFileChooser(@Nullable FileSystemView fsv)
  {
    super(fsv);
    this.directoryPreferenceProperty = null;
  }

  /**
   * Constructs a {@code JFileChooser} using the given current directory
   * and {@code FileSystemView}.
   * @param currentDirectory            current directory, or {@code null} to use the default directory
   * @param fsv                         file system view, or {@code null} for default
   */
  public ResizableFileChooser(@Nullable File currentDirectory,
                              @Nullable FileSystemView fsv)
  {
    super(currentDirectory, fsv);
    this.directoryPreferenceProperty = null;
  }

  /**
   * Constructs a {@code JFileChooser} using the given current directory
   * path and {@code FileSystemView}.
   * @param currentDirectoryPath        path of current directory, or {@code null} to use the default directory
   * @param fsv                         file system view, or {@code null} for default
   */
  public ResizableFileChooser(@Nullable String currentDirectoryPath,
                              @Nullable FileSystemView fsv)
  {
    super(currentDirectoryPath, fsv);
    this.directoryPreferenceProperty = null;
  }

  /**
   * Constructs a {@code JFileChooser} pointing to the user's
   * default directory. This default depends on the operating system.
   * It is typically the "My Documents" folder on Windows, and the
   * user's home directory on Unix.
   * @param directoryPreferenceProperty preference property for storing/restoring the last directory used, or {@code null} to ignore
   */
  public ResizableFileChooser(@Nullable DirectoryPreferenceProperty directoryPreferenceProperty)
  {
    this.directoryPreferenceProperty = directoryPreferenceProperty;
  }

  /**
   * Constructs a {@code JFileChooser} using the given path.
   * Passing in a {@code null}
   * string causes the file chooser to point to the user's default directory.
   * This default depends on the operating system. It is
   * typically the "My Documents" folder on Windows, and the user's
   * home directory on Unix.
   *
   * @param directoryPreferenceProperty preference property for storing/restoring the last directory used, or {@code null} to ignore
   * @param currentDirectoryPath a {@code String} giving the path
   *                             to a file or directory
   */
  public ResizableFileChooser(@Nullable DirectoryPreferenceProperty directoryPreferenceProperty,
                              @Nullable String currentDirectoryPath)
  {
    super(currentDirectoryPath);
    this.directoryPreferenceProperty = directoryPreferenceProperty;
  }

  /**
   * Constructs a {@code JFileChooser} using the given {@code File}
   * as the path. Passing in a {@code null} file
   * causes the file chooser to point to the user's default directory.
   * This default depends on the operating system. It is
   * typically the "My Documents" folder on Windows, and the user's
   * home directory on Unix.
   *
   * @param directoryPreferenceProperty preference property for storing/restoring the last directory used, or {@code null} to ignore
   * @param currentDirectory a {@code File} object specifying
   *                         the path to a file or directory
   */
  public ResizableFileChooser(@Nullable DirectoryPreferenceProperty directoryPreferenceProperty,
                              @Nullable File currentDirectory)
  {
    super(currentDirectory);
    this.directoryPreferenceProperty = directoryPreferenceProperty;
  }

  /**
   * Constructs a {@code JFileChooser} using the given
   * {@code FileSystemView}.
   * @param directoryPreferenceProperty preference property for storing/restoring the last directory used, or {@code null} to ignore
   * @param fsv                         file system view, or {@code null} for default
   */
  public ResizableFileChooser(@Nullable DirectoryPreferenceProperty directoryPreferenceProperty,
                              FileSystemView fsv)
  {
    super(fsv);
    this.directoryPreferenceProperty = directoryPreferenceProperty;
  }

  /**
   * Constructs a {@code JFileChooser} using the given current directory
   * and {@code FileSystemView}.
   * @param directoryPreferenceProperty preference property for storing/restoring the last directory used, or {@code null} to ignore
   * @param currentDirectory            current directory, or {@code null} to use the default directory
   * @param fsv                         file system view, or {@code null} for default
   */
  public ResizableFileChooser(@Nullable DirectoryPreferenceProperty directoryPreferenceProperty,
                              @Nullable File currentDirectory,
                              @Nullable FileSystemView fsv)
  {
    super(currentDirectory, fsv);
    this.directoryPreferenceProperty = directoryPreferenceProperty;
  }

  /**
   * Constructs a {@code JFileChooser} using the given current directory
   * path and {@code FileSystemView}.
   * @param directoryPreferenceProperty preference property for storing/restoring the last directory used, or {@code null} to ignore
   * @param currentDirectoryPath        path of current directory, or {@code null} to use the default directory
   * @param fsv                         file system view, or {@code null} for default
   */
  public ResizableFileChooser(@Nullable DirectoryPreferenceProperty directoryPreferenceProperty,
                              @Nullable String currentDirectoryPath,
                              @Nullable FileSystemView fsv)
  {
    super(currentDirectoryPath, fsv);
    this.directoryPreferenceProperty = directoryPreferenceProperty;
  }

  /**
   * Creates and returns a new {@code JDialog} wrapping
   * {@code this} centered on the {@code parent}
   * in the {@code parent}'s frame.
   * This method can be overriden to further manipulate the dialog,
   * to disable resizing, set the location, etc. Example:
   * <pre>
   *     class MyFileChooser extends JFileChooser {
   *         protected JDialog createDialog(Component parent) throws HeadlessException {
   *             JDialog dialog = super.createDialog(parent);
   *             dialog.setLocation(300, 200);
   *             dialog.setResizable(false);
   *             return dialog;
   *         }
   *     }
   * </pre>
   *
   * @param parent the parent component of the dialog;
   *               can be {@code null}
   * @return a new {@code JDialog} containing this instance
   * @throws java.awt.HeadlessException if GraphicsEnvironment.isHeadless()
   *                                    returns true.
   * @see java.awt.GraphicsEnvironment#isHeadless
   * @since 1.4
   */
  @Override
  protected JDialog createDialog(@Nullable Component parent) throws HeadlessException
  {
    if (directoryPreferenceProperty != null) {
      final File directory = directoryPreferenceProperty.getDirectory();
      if (directory != null) {
        setCurrentDirectory(directory);
      }
    }
    JDialog dialog = super.createDialog(parent);
    PP_FILE_CHOOSER_SIZE.setComponentSize(dialog);
    dialog.setLocationRelativeTo(parent);
    dialog.addComponentListener(SIZE_UPDATER);
    return dialog;
  }

  /**
   * Pops a custom file chooser dialog with a custom approve button.
   *
   * @param parent            the parent component of the dialog;
   *                          can be {@code null}
   * @param approveButtonText the text of the {@code ApproveButton}
   * @return the return state of the file chooser on popdown:
   * <ul>
   * <li>JFileChooser.CANCEL_OPTION
   * <li>JFileChooser.APPROVE_OPTION
   * <li>JFileChooser.ERROR_OPTION if an error occurs or the
   * dialog is dismissed
   * </ul>
   * @throws java.awt.HeadlessException if GraphicsEnvironment.isHeadless()
   *                                    returns true.
   * @see java.awt.GraphicsEnvironment#isHeadless
   */
  @Override
  public int showDialog(@Nullable Component parent, @Nullable String approveButtonText) throws HeadlessException
  {
    int result = super.showDialog(parent, approveButtonText);
    if (result == JFileChooser.APPROVE_OPTION && directoryPreferenceProperty != null) {
      directoryPreferenceProperty.setDirectory(getCurrentDirectory());
    }
    return result;
  }
}
