// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import javax.swing.*;
import java.awt.*;

/**
 * Swing scroll pane which forward the enable status to the component in the viewport.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class PropagateEnableScrollPane
        extends JScrollPane
{
  private static final long serialVersionUID = 6384409329239697702L;

  /**
   * Creates a {@code JScrollPane} that displays the view
   * component in a viewport
   * whose view position can be controlled with a pair of scrollbars.
   * The scrollbar policies specify when the scrollbars are displayed,
   * For example, if {@code vsbPolicy} is
   * {@code VERTICAL_SCROLLBAR_AS_NEEDED}
   * then the vertical scrollbar only appears if the view doesn't fit
   * vertically. The available policy settings are listed at
   * {@link #setVerticalScrollBarPolicy} and
   * {@link #setHorizontalScrollBarPolicy}.
   *
   * @param view      the component to display in the scrollpanes viewport
   * @param vsbPolicy an integer that specifies the vertical
   *                  scrollbar policy
   * @param hsbPolicy an integer that specifies the horizontal
   * @see #setViewportView
   */
  public PropagateEnableScrollPane(Component view, int vsbPolicy, int hsbPolicy)
  {
    super(view, vsbPolicy, hsbPolicy);
  }

  /**
   * Creates a {@code JScrollPane} that displays the
   * contents of the specified
   * component, where both horizontal and vertical scrollbars appear
   * whenever the component's contents are larger than the view.
   *
   * @param view the component to display in the scrollpane's viewport
   * @see #setViewportView
   */
  public PropagateEnableScrollPane(Component view)
  {
    super(view);
  }

  /**
   * Creates an empty (no viewport view) {@code JScrollPane}
   * with specified
   * scrollbar policies. The available policy settings are listed at
   * {@link #setVerticalScrollBarPolicy} and
   * {@link #setHorizontalScrollBarPolicy}.
   *
   * @param vsbPolicy an integer that specifies the vertical
   *                  scrollbar policy
   * @param hsbPolicy an integer that specifies the horizontal
   * @see #setViewportView
   */
  public PropagateEnableScrollPane(int vsbPolicy, int hsbPolicy)
  {
    super(vsbPolicy, hsbPolicy);
  }

  /**
   * Creates an empty (no viewport view) {@code JScrollPane}
   * where both horizontal and vertical scrollbars appear when needed.
   */
  public PropagateEnableScrollPane()
  {
  }

  /**
   * Set whether this is enabled.
   * @param enabled {@code true}: enable component in viewport<br>
   *                {@code false}: disable component in viewport
   */
  @Override
  public void setEnabled(boolean enabled)
  {
    super.setEnabled(enabled);
    Component comp = getViewport().getView();
    if (comp != null) {
      comp.setEnabled(enabled);
    }
  }
}
