// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.i18n.I18n;
import de.caff.util.measure.IllegalPhysicalLengthFormatException;
import de.caff.util.measure.PhysicalLength;

import javax.swing.*;
import javax.swing.text.DefaultFormatter;
import java.text.ParseException;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class PhysicalLengthTextField
        extends JFormattedTextField
{
  private static final long serialVersionUID = 3019826615839351118L;


  /**
   * Constructor.
   */
  public PhysicalLengthTextField()
  {
    super(new PhysicalLengthFormatter());
    FormattingHelper.installValidityHinter(this);
  }

  /**
   * Constructor.
   * @param length initially displayed length
   */
  public PhysicalLengthTextField(@NotNull PhysicalLength length)
  {
    super(new PhysicalLengthFormatter());
    setValue(length);
  }

  /**
   * Get the length displayed in this field.
   * @return length
   */
  public PhysicalLength getLength()
  {
    return (PhysicalLength)getValue();
  }

  /**
   * Formatter implementation for physical lengths.
   */
  public static class PhysicalLengthFormatter
          extends DefaultFormatter
  {
    private static final long serialVersionUID = -8591075710352207274L;

    /**
     * Parses {@code text} returning an arbitrary Object. Some
     * formatters may return null.
     *
     * @param text String to convert
     * @return Object representation of text
     * @throws java.text.ParseException if there is an error in the conversion
     */
    @Override
    public Object stringToValue(String text) throws ParseException
    {
      if (text == null) {
        throw new ParseException("Invalid text!", 0);
      }
      try {
        return PhysicalLength.fromString(text, I18n.getDefaultLocale());
      } catch (IllegalPhysicalLengthFormatException e) {
        ParseException x = new ParseException("Invalid length!", 0);
        x.initCause(e);
        throw x;
      }
    }

    /**
     * Returns the string value to display for {@code value}.
     *
     * @param value Value to convert
     * @return String representation of value
     * @throws java.text.ParseException if there is an error in the conversion
     */
    @Override
    public String valueToString(Object value) throws ParseException
    {
      if (value == null) {
        throw new ParseException("Null value not allowed!", 0);
      }
      return value.toString();
    }

    /**
     * Clone.
     * @return clone
     * @throws CloneNotSupportedException on cloning problems
     */
    @Override
    public PhysicalLengthFormatter clone() throws CloneNotSupportedException
    {
      return (PhysicalLengthFormatter)super.clone();
    }
  }
}
