// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.i18n.I18n;

import javax.swing.filechooser.FileFilter;
import java.io.File;
import java.util.regex.Pattern;

/**
 *  A simple file filter for the Swing file chooser allowing only files with a certain extension.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class MultiFileExtensionFilter
        extends FileFilter
{
  static {
    I18n.addAppResourceBase("de.caff.gimmicks.swing.GimmicksSwingResourceBundle");
  }

  private static final Pattern SEPARATOR = Pattern.compile("\\|");

  /** The description of this filter. */
  private final String description;
  /** The extension to look for. */
  private final String[] extensions;
  /** Allow directories? */
  private boolean allowDirectories = true;

  /**
   *  Create a file filter with a default name.
   *  @param extensions the extensions to look for, separated by <tt>|</tt>
   *  @param description description of the filter
   */
  public MultiFileExtensionFilter(@NotNull String extensions,
                                  @NotNull String description)
  {
    this(SEPARATOR.split(extensions), description);
  }

  /**
   *  Create a file filter with a default name.
   *  @param extensions the extensions to look for
   *  @param description description of the filter
   */
  public MultiFileExtensionFilter(@NotNull String[] extensions, String description)
  {
    this.description = description;
    this.extensions  = new String[extensions.length];
    for (int e = 0;  e < extensions.length;  ++e) {
      if (extensions[e].startsWith(".")) {
        this.extensions[e] = extensions[e].toUpperCase();
      }
      else {
        this.extensions[e] = "." + extensions[e].toUpperCase();
      }
    }
  }

  /**
   * The description of this filter. For example: "JPG and GIF Images"
   *
   * @see javax.swing.filechooser.FileView#getName
   */
  @Override
  public String getDescription()
  {
    return description;
  }

  /**
   * Whether the given file is accepted by this filter.
   */
  @Override
  public boolean accept(File f)
  {
    if (f.isDirectory()) {
      return allowDirectories;
    }
    final String name = f.getName().toUpperCase();
    for (String extension : extensions) {
      if (name.endsWith(extension)) {
        return true;
      }
    }
    return false;
  }

  /**
   *  Are directories allowed by this filter?
   *  @return the answer
   */
  public boolean isAllowDirectories()
  {
    return allowDirectories;
  }

  /**
   *  Set whether directories are allowed by this filter.
   *  @param allowDirectories allow directories?
   */
  public void setAllowDirectories(boolean allowDirectories)
  {
    this.allowDirectories = allowDirectories;
  }
}
