// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.gimmicks.swing;

import de.caff.i18n.I18n;

import javax.swing.*;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import java.awt.*;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.ParsePosition;

/**
 *  Text field, which accepts only double values.
 * <p>
 *  I twill fire property change events for {@link de.caff.gimmicks.swing.JDoubleTextField#PROPERTY_VALUE}
 *  when its value is changed to a valid other value.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class JDoubleTextField
  extends JTextField
{
  /** The default format for doubles. */
  public static final String DEFAULT_FORMAT = "#####0.0#####";
  /** Property for value changes (double). */
  public static final String PROPERTY_VALUE = "JDoubleTextField.VALUE";
  /** Property for validity changes (boolean). */
  public static final String PROPERTY_VALID = "JDoubleTextField.VALID";

  /** The bgcolor to be set when the input is invalid. */
  private static final Color INVALID_BACKGROUND = new Color(0xff, 0x80, 0x80);
  private static final long serialVersionUID = -1887684751695439541L;

  /** The default bg color. */
  private Color  defaultBG;

  /** Is this field valid? */
  private boolean valid = true;

  /** The current value. */
  private double value;

  /** The format for interpreting input values. */
  private DecimalFormat scanFormat; 

  /**
   *  Initialization.
   *  @param  initValue  initialization value
   *  @param  format parse format, see {@link java.text.DecimalFormat}
   */
  private void init(double initValue, String format)
  {
    this.value = initValue;
    scanFormat = new DecimalFormat(format,
				   new DecimalFormatSymbols(I18n.getDefaultLocale()));

    setText(scanFormat.format(initValue));

    defaultBG = getBackground();

    addCaretListener(e -> {
      String text = getText();

      ParsePosition pos = new ParsePosition(0);

      double oldValue = value;
      boolean oldValid = valid;

      Number number = scanFormat.parse(text, pos);
      if (number != null  &&  pos.getIndex() == text.length()) {
        valid = true;
        value = number.doubleValue();
      }
      else {
        valid = false;
      }

      if (oldValid != valid) {
        firePropertyChange(PROPERTY_VALID, oldValid, valid);
      }
      if (valid) {
        setBackground(defaultBG);
        if (value != oldValue) {
          firePropertyChange(PROPERTY_VALUE, oldValue, value);
        }
      }
      else {
        setBackground(INVALID_BACKGROUND);
      }
    });
  }
  
  /**
   *  Create with value == 0.
   */
  public JDoubleTextField()
  {
    this(0);
  }

  /**
   *  Create with the given value.
   *  @param  value start value
   */
  public JDoubleTextField(double value)
  {
    init(value, DEFAULT_FORMAT);
  }

  /**
   *  Create with the given value, using the given format.
   *  @param  value  start value
   *  @param  format format string, compare {@link java.lang.String#format(String, Object...)}
   */
  public JDoubleTextField(double value, String format)
  {
    super();
    if (format == null) {
      format = DEFAULT_FORMAT;
    }
    init(value, format);
  }

  /**
   *  Is this valid?
   *  @return the answer
   */
  public boolean isValidValue() 
  {
    return valid;
  }

  /**
   *  Get the current value.
   *  @return  value, or NaN when content is invalid
   *  @see #isValidValue()
   */
  public double getValue()
  {
    return isValidValue() ? value : Double.NaN;
  }

  /**
   *  Set the value.
   *  @param  value  new value
   */
  public void setValue(double value)
  {
    setText(scanFormat.format(value));
    if (this.value != value) {
      // should have been done by caret listener, but we want to be sure
      double oldValue = this.value;
      this.value = value;
      if (!valid) {
        valid = true;
        firePropertyChange(PROPERTY_VALID, false, true);
      }
      firePropertyChange(PROPERTY_VALUE, oldValue, value);
    }
  }

}
