// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.gimmicks.swing;

import javax.swing.*;
import java.awt.*;

/**
 *  An area showing an image.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class IconCanvasSwing extends JComponent {
  private static final long serialVersionUID = 7269376330156773833L;
  protected ImageIcon    icon;
  protected int margin;

  /**
   *  @param  img   the image to show
   */
  public IconCanvasSwing(Image img) {
    setImage(img);
  }

  /**
   * @param  img   the image to show
   * @param margin margin around image
   */
  public IconCanvasSwing(Image img, int margin) {
    setImage(img);
    this.margin = margin;
  }

  /**
   *  Paint method. Just pint the image.
   *  @param  g   the graphics context
   */
  @Override
  protected void paintComponent(Graphics g)
  {
    if (icon != null) {
      Dimension size = getSize();
      int width  = icon.getIconWidth();
      int height = icon.getIconHeight();

      icon.paintIcon(this, g, (size.width - width)/2, (size.height - height)/2);
    }
  }

  /**
   *  Get the preferred size.
   *  @return preferred size
   */
  @Override
  public Dimension getPreferredSize() {
    if (icon != null) {
      int width  = icon.getIconWidth();
      int height = icon.getIconHeight();
      
      return new Dimension(width + 2*margin, height + 2*margin);
    }
    else {
      return new Dimension(1, 1);
    }
  }

  /**
   *  Get the minimum size
   *  @return the minimum size
   */
  public Dimension getMinimalSize() {
    return getPreferredSize();
  }

  /**
   *  Set the image.
   *  @param  img  new image
   */
  public void setImage(Image img) {
    if (img != null) {
      try {
	// --- be sure image is loaded ---
	MediaTracker mt = new MediaTracker(this);
	mt.addImage(img, 0);
	mt.waitForID(0);
      } catch (InterruptedException x) {
	// do nothing
      }
    }

    icon = new ImageIcon(img);

    // --- set correct size ---
    invalidate();
    repaint();
  }

}
