// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.util.Utility;

import javax.swing.*;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

/**
 * Helper methods for formatting.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class FormattingHelper
{
  /** The value property of a {@link javax.swing.JFormattedTextField}. */
  public static final String PROPERTY_EDIT_VALID = "editValid";

  public static final Color INVALID_BACKGROUND = Utility.getColorParameter("field.invalid.bg",
                                                                           new Color(0xff, 0xe0, 0xe0));

  /** Keep from being initialized. */
  private FormattingHelper()
  {}

  /**
   * Install a visual hinter for validity of a formatted text field.
   * This will make the background light red when the field is invalid.
   * @param textField text field where the hinter is install
   */
  public static void installValidityHinter(@NotNull final JFormattedTextField textField)
  {
    textField.addPropertyChangeListener(PROPERTY_EDIT_VALID,
                                        new PropertyChangeListener()
    {
      private final Color bg =
              textField.isOpaque()
                      ? textField.getBackground()
                      : null;

      @Override
      public void propertyChange(PropertyChangeEvent evt)
      {
        if (textField.isEditValid()) {
          if (bg == null) {
            textField.setOpaque(false);
          }
          else {
            textField.setBackground(bg);
          }
        }
        else {
          if (bg == null) {
            textField.setOpaque(true);
          }
          textField.setBackground(INVALID_BACKGROUND);
        }
      }

    });
    if (!textField.isEditValid()) {
      textField.setOpaque(true);
      textField.setBackground(INVALID_BACKGROUND);
    }
  }

}
