// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.i18n.swing.RJButton;
import de.caff.util.ImprovedStringWriter;
import de.caff.util.Utility;

import javax.swing.*;
import java.awt.*;
import java.awt.datatransfer.*;
import java.io.PrintWriter;

/**
 * Extended error dialog used for debugging.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ExtendedErrorDialog
        extends JDialog
        implements ClipboardOwner
{
  private static final long serialVersionUID = 2767264611137546908L;

  static {
    I18n.addAppResourceBase("de.caff.gimmicks.swing.GimmicksSwingResourceBundle");
  }

  /** Capacity used for all text data. */
  private static final int CAPACITY = 0x1000;
  /** I18n tag for dialog title. */
  static final String I18N_TAG_EXCEPTION_TITLE = "XError#exception#title";
  /** I18n tag for copy to clipboard button. */
  static final String I18N_TAG_BTN_COPY = "XError#button#copy";
  /** I18n tag for close button. */
  static final String I18N_TAG_BTN_CLOSE = "XError#button#close";
  /** I18n tag for copy failure dialog message, %0 is exception. */
  static final String I18N_TAG_ERROR_COPY = "XError!err#copy#msg";
  /** I18n tag for copy failure dialog title. */
  static final String I18N_TAG_TITLE_ERROR_COPY = "XError!err#copy#title";

  /**
   * Constructor.
   * Creates a model dialog.
   *
   * @param owner the owner {@code Frame} from which the dialog is displayed
   *              or {@code null} if this dialog has no owner
   * @param title   dialog title
   * @param message error message
   * @param exception underlying exception, or {@code null}
   * @throws HeadlessException {@code if GraphicsEnvironment.isHeadless()}
   *                           returns {@code true}.
   * @see java.awt.GraphicsEnvironment#isHeadless
   * @see javax.swing.JComponent#getDefaultLocale
   */
  public ExtendedErrorDialog(@Nullable Window owner,
                             @NotNull String title,
                             @NotNull String message,
                             @Nullable Throwable exception)
  {
    super(owner, title, ModalityType.APPLICATION_MODAL);
    setDefaultCloseOperation(DISPOSE_ON_CLOSE);
    final StringBuffer sb = new StringBuffer(CAPACITY);
    sb.append(message).append("\n\n");
    getContentPane().setLayout(new BorderLayout());
    Box north = Box.createVerticalBox();
    for (String line : message.split("\n")) {
      north.add(new JLabel(line, JLabel.LEADING));
    }
    getContentPane().add(north, BorderLayout.NORTH);
    JTextArea area = new JTextArea();
    if (exception != null) {
      String xMessage = exception.getLocalizedMessage();
      if (xMessage == null  ||  xMessage.isEmpty()) {
        xMessage = exception.getMessage();
      }
      ImprovedStringWriter sw = new ImprovedStringWriter(CAPACITY);
      PrintWriter pw = new PrintWriter(sw);
      if (xMessage != null  &&  !xMessage.isEmpty()) {
        pw.write(xMessage);
        pw.write("\n");
      }
      exception.printStackTrace(pw);
      area.setText(sw.getBuffer().toString());
      sb.append("Exception:\n").append(sw.getBuffer().toString());
    }
    JScrollPane scrollPane = new JScrollPane(area);
    scrollPane.setMinimumSize(new Dimension(128, 192));
    scrollPane.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(),
                                                          I18n.getString(I18N_TAG_EXCEPTION_TITLE)));
    getContentPane().add(scrollPane, BorderLayout.CENTER);

    Box south = Box.createHorizontalBox();
    JButton copy = new RJButton(I18N_TAG_BTN_COPY);
    JButton close = new RJButton(I18N_TAG_BTN_CLOSE);
    south.add(copy);
    south.add(Box.createHorizontalGlue());
    south.add(close);

    copy.addActionListener(e -> {
      try {
        Utility.copyTextToClipboard(sb.toString());
      } catch (Throwable x) {
        JOptionPane.showConfirmDialog(getContentPane(),
                                      I18n.format(I18N_TAG_ERROR_COPY, x).split("\n"),
                                      I18n.getString(I18N_TAG_TITLE_ERROR_COPY),
                                      JOptionPane.DEFAULT_OPTION,
                                      JOptionPane.ERROR_MESSAGE);
      }
    });
    close.addActionListener(e -> dispose());
    getContentPane().add(south, BorderLayout.SOUTH);
    pack();
  }

  /**
   * Notifies this object that it is no longer the clipboard owner.
   * This method will be called when another application or another
   * object within this application asserts ownership of the clipboard.
   *
   * @param clipboard the clipboard that is no longer owned
   * @param contents  the contents which this owner had placed on the clipboard
   */
  @Override
  public void lostOwnership(Clipboard clipboard, Transferable contents)
  {
    // ignore
  }
}
