// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.util.swing.SwingHelper;
import de.caff.util.debug.Debug;

import javax.swing.*;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.MissingResourceException;
import java.util.prefs.Preferences;

import static de.caff.gimmicks.swing.CollapsiblePane.*;

/**
 * Collapsible tabbed pane.
 *
 * <p>
 * Although extending JTabbedPane, not all methods are useful.
 * </p>
 * <p>
 * Basically just create this, put it to {@link Side#getNaturalBorder() its natural border}
 * of a container with {@link BorderLayout}, and add some tabs.
 * </p>
 * <p>
 *   If you want to store the expanded sizes to the preferences, you should use one of
 *   the constructors with an id, as that will be used to discern between different
 *   instances of this class used in the same GUI.
 * </p>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since April.28, 2016
 */
public class CollapsibleTabbedPane
        extends JTabbedPane
{
  private static final long serialVersionUID = -6279662510054791683L;

  static {
    I18n.addAppResourceBase("de.caff.gimmicks.swing.GimmicksSwingResourceBundle");
  }

  /** Provide the component dimension based on expand direction. */
  private interface DirectionHelper
  {
    /**
     * Get the expandable dimension.
     * @param basicDimension basic dimension of the component
     * @param expandedSize   size of the expandable direction
     * @return the combined dimension
     */
    @NotNull
    Dimension getDimension(@NotNull Dimension basicDimension,
                           int expandedSize);

    /**
     * Get the coordinate which is expandable.
     * @param x  x coordinate
     * @param y  y coordinate
     * @return either x or y
     */
    int getCoordinateOfInterest(int x, int y);
  }
  /** Direction in which a {@link CollapsiblePane} is expandable. */
  public enum ExpandableDirection
    implements DirectionHelper
  {
    /** Can expand in horizontal direction. */
    Horizontally {
      @NotNull
      @Override
      public Dimension getDimension(@NotNull Dimension basicDimension, int expandedSize)
      {
        return new Dimension(expandedSize, basicDimension.height);
      }

      @Override
      public int getCoordinateOfInterest(int x, int y)
      {
        return x;
      }
    },
    /** Can expand in vertical direction. */
    Vertically {
      @NotNull
      @Override
      public Dimension getDimension(@NotNull Dimension basicDimension, int expandedSize)
      {
        return new Dimension(basicDimension.width, expandedSize);
      }

      @Override
      public int getCoordinateOfInterest(int x, int y)
      {
        return y;
      }
    }
  }

  /**
   * Set a divider.
   */
  private interface DividerHandler
  {
    /**
     * Create a divider by setting a border to the component.
     * @param comp        component to instrument with a divider
     * @param dividerSize size of divider
     * @param color       color of divider, {@code null} for empty
     */
    void setDivider(@NotNull JComponent comp,
                    int dividerSize,
                    @Nullable Color color);

    /**
     * Is the cursor over the divider set by {@link #setDivider(JComponent, int, Color)}?
     * @param comp        component with divider
     * @param pos         mouse position
     * @param dividerSize size of divider
     * @return {@code true} if the mouse is over the divider,<br>
     *         {@code false} if not
     */
    boolean isOverDivider(@NotNull JComponent comp,
                          @NotNull Point pos,
                          int dividerSize);

    /**
     * Get the new size during resizing.
     * @param comp        component with divider
     * @param pos         mouse position
     * @param dividerSize dimension indicated by mouse position
     * @return resizred dimension
     */
    @NotNull
    Dimension getSize(@NotNull JComponent comp,
                      @NotNull Point pos,
                      int dividerSize);
  }

  /**
   *  Helper class which sets a divider by setting a matte or empty border
   *  woth given insets.
   */
  private abstract static class InsetsDividerHandler
    implements DividerHandler
  {
    /**
     * Set the divider by setting a border with the given insets.
     * @param comp   component to get border
     * @param top    top size of border
     * @param left   left size of border
     * @param bottom bottom size of border
     * @param right  right size of border
     * @param color  color of border, {@code null} for empty
     */
    protected void setDivider(@NotNull JComponent comp,
                              int top, int left, int bottom, int right,
                              @Nullable Color color)
    {
      if (color != null) {
        comp.setBorder(BorderFactory.createMatteBorder(top, left, bottom, right, color));
      }
      else {
        comp.setBorder(BorderFactory.createEmptyBorder(top, left, bottom, right));
      }
    }
  }

  /** The side where the expandable pane is placed. */
  public enum Side
          implements DirectionHelper,
                     DividerHandler
  {
    /** At the northern side. */
    North(SwingConstants.TOP,
          BorderLayout.NORTH,
          Cursor.S_RESIZE_CURSOR,
          ExpandableDirection.Vertically,
          I18N_TAG_ACTION_HIDE_NORTH,
          new InsetsDividerHandler()
          {
            @Override
            public void setDivider(@NotNull JComponent comp, int dividerSize, @Nullable Color color)
            {
              setDivider(comp, 0, 0, dividerSize, 0, color);
            }

            @Override
            public boolean isOverDivider(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
            {
              return comp.getHeight() - pos.y < dividerSize;
            }

            @NotNull
            @Override
            public Dimension getSize(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
            {
              return new Dimension(comp.getWidth(),
                                   pos.y + dividerSize/2);
            }
          }),
    /** At the eastern side. */
    East(SwingConstants.RIGHT,
         BorderLayout.EAST,
         Cursor.W_RESIZE_CURSOR,
         ExpandableDirection.Horizontally,
         I18N_TAG_ACTION_HIDE_EAST,
         new InsetsDividerHandler()
         {
           @Override
           public void setDivider(@NotNull JComponent comp, int dividerSize, @Nullable Color color)
           {
             setDivider(comp, 0, dividerSize, 0, 0, color);
           }

           @Override
           public boolean isOverDivider(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
           {
             return pos.x < dividerSize;
           }

           @NotNull
           @Override
           public Dimension getSize(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
           {
             return new Dimension(comp.getWidth() - pos.x -dividerSize/2,
                                  comp.getHeight());
           }
         }),
    /** At the southern side. */
    South(SwingConstants.BOTTOM,
          BorderLayout.SOUTH,
          Cursor.N_RESIZE_CURSOR,
          ExpandableDirection.Vertically,
          I18N_TAG_ACTION_HIDE_SOUTH,
          new InsetsDividerHandler()
          {
            @Override
            public void setDivider(@NotNull JComponent comp, int dividerSize, @Nullable Color color)
            {
              setDivider(comp, dividerSize, 0, 0, 0, color);
            }

            @Override
            public boolean isOverDivider(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
            {
              return pos.y < dividerSize;
            }

            @NotNull
            @Override
            public Dimension getSize(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
            {
              return new Dimension(comp.getWidth(),
                                   comp.getHeight() - pos.y - dividerSize/2);
            }
          }),
    /** At the western side. */
    West(SwingConstants.LEFT,
         BorderLayout.WEST,
         Cursor.E_RESIZE_CURSOR,
         ExpandableDirection.Horizontally,
         I18N_TAG_ACTION_HIDE_WEST,
         new InsetsDividerHandler()
         {
           @Override
           public void setDivider(@NotNull JComponent comp, int dividerSize, @Nullable Color color)
           {
             setDivider(comp, 0, 0, 0, dividerSize, color);
           }

           @Override
           public boolean isOverDivider(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
           {
             return comp.getWidth() - pos.x < dividerSize;
           }

           @NotNull
           @Override
           public Dimension getSize(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
           {
             return new Dimension(pos.x + dividerSize / 2,
                                  comp.getHeight());
           }
         });

    final int tabPlacement;
    @NotNull
    final String naturalBorder;
    @NotNull
    final String i18nHideActionTag;
    @NotNull
    private final ExpandableDirection expandableDirection;
    @NotNull
    private final DividerHandler dividerHandler;
    @NotNull
    private final Cursor resizeCursor;


    Side(int swingConstannt,
         @NotNull String naturalBorder,
         int cursor,
         @NotNull ExpandableDirection expandableDirection,
         @NotNull String tag,
         @NotNull DividerHandler dividerHandler)
    {
      this.tabPlacement = swingConstannt;
      this.naturalBorder = naturalBorder;
      this.expandableDirection = expandableDirection;
      i18nHideActionTag = tag;
      this.dividerHandler = dividerHandler;
      resizeCursor = Cursor.getPredefinedCursor(cursor);
    }

    /**
     * Get the tab placement constant which is associated with this side.
     * @return tab placement constant
     */
    public int getTabPlacement()
    {
      return tabPlacement;
    }

    /**
     * Get the natural border for {@link BorderLayout} where a collapsible
     * tabbed pane for this size should be placed.
     * @return border layout border constant
     */
    @NotNull
    public String getNaturalBorder()
    {
      return naturalBorder;
    }

    /**
     * Get the i18n action tag for the hide action.
     * @return i18n tag
     */
    @NotNull
    public String getI18nHideActionTag()
    {
      return i18nHideActionTag;
    }

    /**
     * Get the direction in which a collapsible tabbed pane for this side
     * is expanded.
     * @return expandable direction
     */
    @NotNull
    public ExpandableDirection getExpandableDirection()
    {
      return expandableDirection;
    }

    /**
     * Get the resize cursor associated with this size.
     * @return resize cursor
     */
    @NotNull
    public Cursor getResizeCursor()
    {
      return resizeCursor;
    }

    @NotNull
    @Override
    public Dimension getDimension(@NotNull Dimension basicDimension, int expandedSize)
    {
      return expandableDirection.getDimension(basicDimension, expandedSize);
    }

    @Override
    public int getCoordinateOfInterest(int x, int y)
    {
      return expandableDirection.getCoordinateOfInterest(x, y);
    }

    @Override
    public void setDivider(@NotNull JComponent comp, int dividerSize, @Nullable Color color)
    {
      dividerHandler.setDivider(comp, dividerSize, color);
    }

    @Override
    public boolean isOverDivider(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
    {
      return dividerHandler.isOverDivider(comp, pos, dividerSize);
    }

    @NotNull
    @Override
    public Dimension getSize(@NotNull JComponent comp, @NotNull Point pos, int dividerSize)
    {
      return dividerHandler.getSize(comp, pos, dividerSize);
    }
  }

  @NotNull
  final Side side;
  /** ID used for preferences. */
  @NotNull
  final String id;
  @Nullable
  private Preferences preferences;

  /**
   * Create a collapsible tabbed pane.
   * This uses {@link #WRAP_TAB_LAYOUT wrapped tab layout} and a generic id which is
   * shared with all instances of this class..
   * @param side side to which this pane is added
   */
  public CollapsibleTabbedPane(@NotNull Side side)
  {
    this(side, WRAP_TAB_LAYOUT);
  }

  /**
   * Constructor.
   * This uses a generic id ahared wiht all instances of this class.
   * @param side side to which this pane is added
   * @param tabLayoutPolicy  tab layout policy
   */
  public CollapsibleTabbedPane(@NotNull Side side, int tabLayoutPolicy)
  {
    this("COLL_TAB_PANE", side, tabLayoutPolicy);
  }

  /**
   * Constructor.
   * This uses {@link #WRAP_TAB_LAYOUT wrapped tab layout}.
   * @param id   unique id used for storing and recovering preferences
   * @param side side to which this pane is added
   */
  public CollapsibleTabbedPane(@NotNull String id,
                               @NotNull Side side)
  {
    this(id, side, WRAP_TAB_LAYOUT);
  }

  /**
   * Constructor.
   * @param id   unique id used for storing and recovering preferences
   * @param side side to which this pane is added
   * @param tabLayoutPolicy  tab layout policy, see extended class
   */
  public CollapsibleTabbedPane(@NotNull String id,
                               @NotNull Side side,
                               int tabLayoutPolicy)
  {
    super(side.tabPlacement,
          tabLayoutPolicy);
    this.id = id;
    this.side = side;
    addMouseListener(new MouseListener()
    {
      @Override
      public void mouseClicked(MouseEvent e)
      {
        if (e.getButton() != MouseEvent.BUTTON1  ||
            e.getClickCount() != 2) {
          return;
        }
        int index = indexAtLocation(e.getX(), e.getY());
        if (index >= 0  &&  index == getSelectedIndex()) {
          setSelectedIndex(-1);
        }
      }

      @Override
      public void mousePressed(MouseEvent e)
      {
      }

      @Override
      public void mouseReleased(MouseEvent e)
      {
      }

      @Override
      public void mouseEntered(MouseEvent e)
      {
      }

      @Override
      public void mouseExited(MouseEvent e)
      {
      }
    });
  }

  /**
   * Get the side for which this pane is configured.
   * @return side to which this pane should be attached
   */
  @NotNull
  public Side getSide()
  {
    return side;
  }

  /**
   * Get the id of this pane.
   * @return id
   */
  @NotNull
  public String getId()
  {
    return id;
  }

  /**
   * Add a tab from an i18n resource tag.
   * @param baseTag base tag of i18n resource
   * @param comp    component to insert
   */
  public void addTabFromResource(@NotNull String baseTag,
                                 @NotNull JComponent comp)
  {
    insertTabFromResource(baseTag, comp, getTabCount());
  }

  /**
   * Insert a tab from an i18n resource tag.
   * @param baseTag base tag of i18n resource
   * @param comp    component to insert
   * @param index   index where to insert
   */
  public void insertTabFromResource(@NotNull String baseTag,
                                    @NotNull JComponent comp,
                                    int index)
  {
    insertTab(baseTag,
              getText(baseTag),
              getIcon(baseTag),
              comp,
              getTooltip(baseTag),
              index);
  }

  @Nullable
  private static String getI18nResourceOrNull(@NotNull String tag)
  {
    try {
      return I18n.getString(tag);
    } catch (MissingResourceException x) {
      return null;
    }
  }

  @Nullable
  private static Icon getIcon(@NotNull String baseTag)
  {
    String path = getI18nResourceOrNull(baseTag + I18n.SUFFIX_ICON);
    if (path != null) {
      return SwingHelper.loadIconResource(path);
    }
    return null;
  }

  @Nullable
  private static String getText(@NotNull String baseTag)
  {
    return getI18nResourceOrNull(baseTag + I18n.SUFFIX_TEXT);
  }

  @Nullable
  private static String getTooltip(@NotNull String baseTag)
  {
    return getI18nResourceOrNull(baseTag + I18n.SUFFIX_TOOLTIP);
  }


  /**
   * Inserts a new tab for the given component, at the given index,
   * represented by the given title and/or icon, either of which may
   * be {@code null}.
   *
   * @param title the title to be displayed on the tab
   * @param icon the icon to be displayed on the tab
   * @param component the component to be displayed when this tab is clicked.
   * @param tip the tooltip to be displayed for this tab
   * @param index the position to insert this new tab
   *       ({@code &gt; 0 and &lt;= getTabCount()})
   *
   * @throws IndexOutOfBoundsException if the index is out of range
   *         ({@code &lt; 0 or &gt; getTabCount()})
   *
   * @see #addTab
   * @see #removeTabAt
   */
  @Override
  public void insertTab(String title, Icon icon, Component component, String tip, int index)
  {
    insertTab(title == null ? Integer.toString(index) : title,
              title, icon, component, tip, index);
  }

  /**
   * Inserts a new tab for the given component, at the given index,
   * represented by the given title and/or icon, either of which may
   * be {@code null}.
   *
   * @param id    tab ID
   * @param title the title to be displayed on the tab
   * @param icon the icon to be displayed on the tab
   * @param component the component to be displayed when this tab is clicked.
   * @param tip the tooltip to be displayed for this tab
   * @param index the position to insert this new tab
   *       ({@code &gt; 0 and &lt;= getTabCount()})
   *
   * @throws IndexOutOfBoundsException if the index is out of range
   *         ({@code &lt; 0 or &gt; getTabCount()})
   *
   * @see #addTab
   * @see #removeTabAt
   */
  public void insertTab(@NotNull String id,
                        @Nullable String title,
                        @Nullable Icon icon,
                        @NotNull Component component,
                        @Nullable String tip,
                        final int index) {
    if (title == null) {
      title = "";
    }
    final CollapsiblePane collapse = new CollapsiblePane(this, id, title, (JComponent)component);
    super.insertTab(title, icon,
                    collapse,
                    tip, index);
    if (preferences != null) {
      SwingUtilities.invokeLater(() -> collapse.loadPreferences(preferences));
    }
    switch (side) {
    case North:
      // standard is okay
      break;
    case East:
      JLabel label = new JLabel(title, icon, LEADING) {
        private static final long serialVersionUID = -6373582197823167286L;

        @Override
        public void updateUI()
        {
          setUI(new RotatedLabelUI(false));
        }
      };
      label.setUI(new RotatedLabelUI(false));
      setTabComponentAt(index, label);
      break;
    case South:
      break;
    case West:
      label = new JLabel(title, icon, LEADING) {
        private static final long serialVersionUID = -3197594686535605227L;

        @Override
        public void updateUI()
        {
          setUI(new RotatedLabelUI(true));
        }
      };
      label.setUI(new RotatedLabelUI(true));
      setTabComponentAt(index, label);
      break;
    }
  }

  /**
   * Load the expanded sizes of the tabs from the given preference.
   * @param preferences preferences
   */
  public void loadPreferences(@NotNull Preferences preferences)
  {
    this.preferences = preferences;
    for (int i = getTabCount() - 1;  i >= 0;  --i) {
      try {
        CollapsiblePane cp = (CollapsiblePane)getComponentAt(i);
        cp.loadPreferences(preferences);
      } catch (ClassCastException e) {
        // could happen if Swing implementation changes and calling
        // insertTab() could be avoided when adding components
        Debug.error(e);
      }
    }
  }

  /**
   * Store the expanded sizes of the tabs to the given preferences.
   * @param preferences preferences
   */
  public void storePreferences(@NotNull Preferences preferences)
  {
    for (int i = getTabCount() - 1;  i >= 0;  --i) {
      try {
        CollapsiblePane cp = (CollapsiblePane)getComponentAt(i);
        cp.storePreferences(preferences);
      } catch (ClassCastException e) {
        // could happen if Swing implementation changes and calling
        // insertTab() could be avoided when adding components
        Debug.error(e);
      }
    }
  }

  @Override
  public int indexOfComponent(Component component)
  {
    for (int i = getTabCount() - 1;  i >= 0;  --i) {
      try {
        CollapsiblePane cp = (CollapsiblePane)getComponentAt(i);
        if (cp == component) {
          // happens whne called inside class
          return i;
        }
        if (cp.getInternalComponent() == component) {
          // happens when called from outside class
          return i;
        }
      } catch (ClassCastException e) {
        // could happen if Swing implementation changes and calling
        // insertTab() could be avoided when adding components
        Debug.error(e);
      }
    }
    return -1;
  }

  @Override
  public void setSelectedIndex(int index)
  {
    super.setSelectedIndex(index);
    for (int i = getTabCount() - 1;  i >= 0;  --i) {
      ((CollapsiblePane)getComponentAt(i)).setExpanded(i == index);
    }
  }
}
