// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.debug.Debug;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.prefs.PreferenceChangeEvent;
import java.util.prefs.PreferenceChangeListener;
import java.util.prefs.Preferences;

/**
 * The collapsible content of a {@link CollapsibleTabbedPane}.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since April.28, 2016
 */
public class CollapsiblePane
        extends JPanel
        implements PreferenceChangeListener
{
  private static final long serialVersionUID = 8173423979719285623L;

  private static final int DIVIDER_SIZE = 8;

  static final String I18N_TAG_ACTION_HIDE_NORTH = "CollapsiblePane.Hide.North";
  static final String I18N_TAG_ACTION_HIDE_EAST  = "CollapsiblePane.Hide.East";
  static final String I18N_TAG_ACTION_HIDE_SOUTH = "CollapsiblePane.Hide.South";
  static final String I18N_TAG_ACTION_HIDE_WEST  = "CollapsiblePane.Hide.West";

  @NotNull
  private final CollapsibleTabbedPane tabbed;
  @NotNull
  private final JLabel titleLabel;
  @NotNull
  private final Box titleBar;
  @NotNull
  private final JComponent internalComponent;
  private int expandedSize = -1;
  private boolean expanded = true;
  @Nullable
  private Preferences preferences;
  @NotNull
  private final String preferenceKey;
  /** In the beginning... */
  private boolean initialising = true;

  /**
   * Constructor.
   * @param tabbed tabbed pane to which this pane belongs
   * @param id     id for preference access
   * @param title  title of pane
   * @param internalComponent component which is handle by this collapsible pane
   */
  CollapsiblePane(@NotNull final CollapsibleTabbedPane tabbed,
                  @NotNull String id,
                  @NotNull String title,
                  @NotNull JComponent internalComponent)
  {
    super(new BorderLayout());
    preferenceKey = id;
    this.tabbed = tabbed;
    ResourcedAction hideAction = new ResourcedAction(tabbed.side.i18nHideActionTag)
    {
      private static final long serialVersionUID = 3014693439018404326L;

      @Override
      public void actionPerformed(ActionEvent e)
      {
        tabbed.setSelectedIndex(-1);
        tabbed.invalidate();
        tabbed.repaint();
      }
    };
    titleBar = Box.createHorizontalBox();
    final JButton hideButtonStart = new JButton(hideAction);
    hideButtonStart.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
    titleBar.add(hideButtonStart);
    titleLabel = new JLabel(title);
    titleBar.add(titleLabel);
    titleBar.add(Box.createHorizontalGlue());
    final JButton hideButtonEnd = new JButton(hideAction);
    hideButtonEnd.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
    titleBar.add(hideButtonEnd);
    titleBar.setBorder(BorderFactory.createMatteBorder(0, 0, 2, 0, Color.black)); // todo: use UI color
    add(titleBar, BorderLayout.NORTH);
    add(internalComponent, BorderLayout.CENTER);
    this.internalComponent = internalComponent;
    setMinimumSize(new Dimension(0, 0));
    setExpanded(false);
    setDividerActive(false, false);

    final MouseAdapter mouseHandler = new MouseAdapter()
    {
      private Boolean hasDragged;
      @Override
      public void mousePressed(MouseEvent e)
      {
        if (e.getButton() == MouseEvent.BUTTON1 &&
            tabbed.side.isOverDivider(CollapsiblePane.this, e.getPoint(), DIVIDER_SIZE)) {
          hasDragged = false;
          setDividerActive(true, true);
        }
      }

      @Override
      public void mouseReleased(MouseEvent e)
      {
        if (hasDragged != null  &&  hasDragged  && e.getButton() == MouseEvent.BUTTON1) {
          updateSize(e.getPoint(), true);
        }
        hasDragged = null;
        setDividerActive(e.getX() < DIVIDER_SIZE, false);
      }

      @Override
      public void mouseEntered(MouseEvent e)
      {
        mouseMoved(e);
      }

      @Override
      public void mouseMoved(MouseEvent e)
      {
        setDividerActive(tabbed.side.isOverDivider(CollapsiblePane.this,
                                                   e.getPoint(),
                                                   DIVIDER_SIZE),
                         hasDragged != null);
      }

      @Override
      public void mouseExited(MouseEvent e)
      {
        setDividerActive(false, hasDragged != null);
      }

      @Override
      public void mouseDragged(MouseEvent e)
      {
        if (hasDragged != null) {
          hasDragged = true;
          updateSize(e.getPoint(), false);
        }
      }

      void updateSize(@NotNull Point pos, boolean finished)
      {
        Dimension dim = tabbed.side.getSize(CollapsiblePane.this,
                                            pos,
                                            DIVIDER_SIZE);
        dim = getRestrictedDimension(dim, 0.9);
        setPreferredSize(dim);
        setSize(dim);
        tabbed.revalidate();
        tabbed.repaint();
        expandedSize = tabbed.side.getCoordinateOfInterest(dim.width, dim.height);
        if (finished && preferences != null) {
          storePreferences(preferences);
        }
      }
    };
    addMouseListener(mouseHandler);
    addMouseMotionListener(mouseHandler);
  }

  /**
   * Get the internal component.
   * @return internal component
   */
  @NotNull
  public JComponent getInternalComponent()
  {
    return internalComponent;
  }

  private void setDividerActive(boolean active, boolean dragging)
  {
    if (dragging) {
      setCursor(tabbed.side.getResizeCursor());
      tabbed.side.setDivider(this, DIVIDER_SIZE, UIManager.getColor("MenuItem.selectionBackground"));

    }
    else if (active) {
      setCursor(tabbed.side.getResizeCursor());
      tabbed.side.setDivider(this, DIVIDER_SIZE, UIManager.getColor("TabbedPane.highlight"));
    }
    else{
      setCursor(Cursor.getDefaultCursor());
      tabbed.side.setDivider(this, DIVIDER_SIZE, null);
    }
  }

  @Override
  public void addNotify()
  {
    super.addNotify();
  }

  @Override
  public Dimension getPreferredSize()
  {
    if (expanded) {
      Dimension preferredSize = super.getPreferredSize();
      if (initialising) {
        preferredSize = getRestrictedDimension(preferredSize, 0.5);
        initialising = false;
      }
      return preferredSize;
    }
    else {
      return tabbed.side.getDimension(super.getPreferredSize(), 0);
    }
  }

  /**
   * Restrict a dimension to the grand parents size.
   * @param dim  dimension to restrict
   * @param part maximum part of relevant dimension
   * @return restricted dimension
   */
  @NotNull
  private Dimension getRestrictedDimension(@NotNull Dimension dim,
                                           double part)
  {
    Component grandParent = tabbed.getParent();
    if (grandParent != null) {
      // avoid tab becoming so large it hides the GUI including its divider
      Dimension grandParentSize = grandParent.getSize();
      final int size = tabbed.side.getCoordinateOfInterest(grandParentSize.width,
                                                           grandParentSize.height);
      final int pref = tabbed.side.getCoordinateOfInterest(dim.width,
                                                           dim.height);
      if (pref < DIVIDER_SIZE || pref > (int)(size*part) - DIVIDER_SIZE) {
        // set dedicated preferred size
        if (!initialising && pref < 2*DIVIDER_SIZE) {
          // too small during resize
          dim = tabbed.side.getDimension(dim,
                                         2*DIVIDER_SIZE);
        }
        else {
          // initially make it large in both cases
          dim = tabbed.side.getDimension(dim,
                                         (int)(size*part) - DIVIDER_SIZE);
        }
        setPreferredSize(dim);
      }
    }
    return dim;
  }

  @Override
  public Dimension getSize()
  {
    if (expanded) {
      return super.getSize();
    }
    else {
      return tabbed.side.getDimension(super.getSize(), 0);
    }
  }

  @Override
  public Dimension getMinimumSize()
  {
    if (expanded) {
      return super.getMinimumSize();
    }
    else {
      return tabbed.side.getDimension(super.getMinimumSize(), 0);
    }
  }

  void setExpanded(boolean aFlag)
  {
    if (expanded == aFlag) {
      return;
    }
    expanded = aFlag;
    boolean isVisible = isVisible();
    if (isVisible) {
      expandedSize = tabbed.side.getCoordinateOfInterest(getWidth(), getHeight());
    }
    else {
      if (aFlag) {
        setSize(tabbed.side.getDimension(getSize(), expandedSize));
      }
    }
    setVisible(aFlag);
    if (!aFlag) {
      setSize(tabbed.side.getDimension(getSize(), 0));
      //tabbed.setSelectedIndex(-1);
    }
    tabbed.revalidate();
  }

  /**
   * Get the preference key.
   * @return preference key
   */
  @NotNull
  public String getPreferenceKey()
  {
    return preferenceKey;
  }

  /**
   * Get the expanded size preference key.
   * @return key
   */
  @NotNull
  private String getExpandedSizeKey()
  {
    return String.format("%s.%s.expandedSize", tabbed.id, preferenceKey);
  }

  @Override
  public void removeNotify()
  {
    super.removeNotify();
    if (preferences != null)  {
      preferences.removePreferenceChangeListener(this);
    }
  }

  /**
   * Load the expanded size from the preferences.
   * @param preferences preferences
   */
  public void loadPreferences(@NotNull Preferences preferences)
  {
    final int size = preferences.getInt(getExpandedSizeKey(), -1);
    if (size > 0) {
      expandedSize = size;
      final Dimension dim = tabbed.side.getDimension(getSize(),
                                                     expandedSize);
      setPreferredSize(dim);
      setSize(dim);
      tabbed.revalidate();
      tabbed.repaint();
    }
    if (this.preferences != null)  {
      this.preferences.removePreferenceChangeListener(this);
    }
    preferences.addPreferenceChangeListener(this);
    this.preferences = preferences;
  }

  /**
   * Store the expanded size to the preferences.
   * @param preferences preferences
   */
  public void storePreferences(@NotNull Preferences preferences)
  {
    if (expandedSize > 0) {
      preferences.putInt(getExpandedSizeKey(), expandedSize);
    }
  }

  @Override
  public void preferenceChange(PreferenceChangeEvent evt)
  {
    if (getExpandedSizeKey().equals(evt.getKey())) {
      try {
        int size = Integer.parseInt(evt.getNewValue());
        if (size != expandedSize) {
          expandedSize = size;
        }
      } catch (NumberFormatException e) {
        Debug.error(e);
      }
    }
  }
}
