// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2013-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;
import de.caff.i18n.I18n;

import javax.swing.*;
import javax.swing.plaf.basic.BasicButtonUI;
import java.awt.*;
import java.awt.event.*;

/**
 * A tabbed pane which adds close buttons to its tabs.
 * <p>
 * It wraps the default tab component with a panel showing an additional close button.
 * Therefore the result of {@link #getTabComponentAt(int)} differs from the component
 * added via {@link #setTabComponentAt(int, java.awt.Component)}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class CloseableTabbedPane
        extends JTabbedPane
{
  private static final long serialVersionUID = -700158248486815620L;

  static {
    I18n.addAppResourceBase("de.caff.gimmicks.swing.GimmicksSwingResourceBundle");
  }

  /**
   * Creates an empty {@code TabbedPane} with a default
   * tab placement of {@code JTabbedPane.TOP}.
   *
   * @see #addTab
   */
  public CloseableTabbedPane()
  {
  }

  /**
   * Creates an empty {@code TabbedPane} with the specified tab placement
   * of either: {@code JTabbedPane.TOP}, {@code JTabbedPane.BOTTOM},
   * {@code JTabbedPane.LEFT}, or {@code JTabbedPane.RIGHT}.
   *
   * @param tabPlacement the placement for the tabs relative to the content
   * @see #addTab
   */
  public CloseableTabbedPane(int tabPlacement)
  {
    super(tabPlacement);
  }

  /**
   * Creates an empty {@code TabbedPane} with the specified tab placement
   * and tab layout policy.  Tab placement may be either:
   * {@code JTabbedPane.TOP}, {@code JTabbedPane.BOTTOM},
   * {@code JTabbedPane.LEFT}, or {@code JTabbedPane.RIGHT}.
   * Tab layout policy may be either: {@code JTabbedPane.WRAP_TAB_LAYOUT}
   * or {@code JTabbedPane.SCROLL_TAB_LAYOUT}.
   *
   * @param tabPlacement    the placement for the tabs relative to the content
   * @param tabLayoutPolicy the policy for laying out tabs when all tabs will not fit on one run
   * @throws IllegalArgumentException
   *          if tab placement or tab layout policy are not
   *          one of the above supported values
   * @see #addTab
   * @since 1.4
   */
  public CloseableTabbedPane(int tabPlacement, int tabLayoutPolicy)
  {
    super(tabPlacement, tabLayoutPolicy);
  }

  /**
   * Inserts a {@code component}, at {@code index},
   * represented by a {@code title} and/or {@code icon},
   * either of which may be {@code null}.
   * Uses java.util.Vector internally, see {@code insertElementAt}
   * for details of insertion conventions.
   *
   * @param title     the title to be displayed in this tab
   * @param icon      the icon to be displayed in this tab
   * @param component The component to be displayed when this tab is clicked.
   * @param tip       the tooltip to be displayed for this tab
   * @param index     the position to insert this new tab
   * @see #addTab
   * @see #removeTabAt
   */
  @Override
  public void insertTab(String title, Icon icon, Component component, String tip, int index)
  {
    super.insertTab(title, icon, component, tip, index);
    setTabComponentAt(index, new JLabel()
    {
      private static final long serialVersionUID = 3727051960111348866L;

      /**
       * Returns the text string that the label displays.
       *
       * @return a String
       * @see #setText
       */
      @Override
      public String getText()
      {
        final int i = indexOfTabComponent(this);
        return i >= 0 ? getTitleAt(i) : null;
      }

      /**
       * Returns the graphic image (glyph, icon) that the label displays.
       *
       * @return an Icon
       * @see #setIcon
       */
      @Override
      public Icon getIcon()
      {
        final int i = indexOfTabComponent(this);
        return i >= 0 ? getIconAt(i) : null;
      }
    });
  }

  /**
   * Sets the component that is responsible for rendering the
   * title for the specified tab.  A null value means
   * {@code JTabbedPane} will render the title and/or icon for
   * the specified tab.  A non-null value means the component will
   * render the title and {@code JTabbedPane} will not render
   * the title and/or icon.
   * <p>
   * Note: The component must not be one that the developer has
   * already added to the tabbed pane.
   * <p>
   * Attention: Because of the internal implementation which adds the
   * close button the component returned by {@link #getTabComponentAt(int)}
   * is not the one added here, but its first child.
   *
   * @param index     the tab index where the component should be set
   * @param component the component to render the title for the
   *                  specified tab
   * @throws IndexOutOfBoundsException if index is out of range
   *                                   (index &lt; 0 || index &gt;= tab count)
   * @throws IllegalArgumentException  if component has already been
   *                                   added to this {@code JTabbedPane}
   * @see #getTabComponentAt
   * @since 1.6
   */
  @Override
  public void setTabComponentAt(int index, Component component)
  {
    super.setTabComponentAt(index, component == null ? null : new CloseableTab(component));
  }

  /**
   * Returns the index of the tab for the specified tab component.
   * Returns -1 if there is no tab for this tab component.
   *
   * @param tabComponent the tab component for the tab
   * @return the first tab which matches this tab component, or -1
   *         if there is no tab for this tab component
   * @see #setTabComponentAt
   * @see #getTabComponentAt
   * @since 1.6
   */
  @Override
  public int indexOfTabComponent(Component tabComponent)
  {
    if (tabComponent == null || tabComponent instanceof CloseableTab) {
      return super.indexOfTabComponent(tabComponent);
    }
    Component parent = tabComponent.getParent();
    if (parent != null &&  parent instanceof CloseableTab) {
      return super.indexOfTabComponent(parent);
    }
    return super.indexOfTabComponent(tabComponent);
  }

  /**
   * Closeable tab class.
   */
  private class CloseableTab extends JPanel
  {
    private static final long serialVersionUID = -5447529075262040118L;

    /**
     * Constructor.
     */
    public CloseableTab(@NotNull Component component)
    {
      setLayout(new BorderLayout(4, 0));
      add(component, BorderLayout.CENTER);

      JButton closeButton = new CloseButton();
      closeButton.addActionListener(e -> {
        final int tabIndex = indexOfTabComponent(CloseableTab.this);
        if (tabIndex >= 0) {
          removeTabAt(tabIndex);
        }
      });
      add(closeButton, BorderLayout.EAST);

      setOpaque(false); // don't fill background
    }

    /**
     * Get the wrapped component.
     * @return wrapped component
     */
    public Component getComponent()
    {
      return getComponent(0);
    }
  }

  /**
   * Listener setting the rollOver property of the close button.
   */
  private static final MouseListener ROLLOVER_LISTENER = new MouseAdapter()
  {
    /**
     * {@inheritDoc}
     */
    @Override
    public void mouseEntered(MouseEvent e)
    {
      Component c = e.getComponent();
      if (c instanceof CloseButton) {
        ((CloseButton)c).setRollOver(true);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void mouseExited(MouseEvent e)
    {
      Component c = e.getComponent();
      if (c instanceof CloseButton) {
        ((CloseButton)c).setRollOver(false);
      }
    }
  };

  private static final Stroke CROSS_STROKE = new BasicStroke(2, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER);
  private static final Color  ACTIVE_FG_COLOR = Color.white;
  private static final Color  ACTIVE_BG_COLOR = new Color(0xc0, 0x00, 0x00);
  private static final Color  DARKER_BG_COLOR = ACTIVE_BG_COLOR.darker();
  private static final Color  LIGHTER_BG_COLOR = ACTIVE_BG_COLOR.brighter();

  /**
   * Close button helper class.
   */
  private static class CloseButton extends JButton
  {
    private static final long serialVersionUID = -1353081873873986435L;
    private static final int BORDER = 2;
    private static final int CROSS_INSET = 4;
    private static final int SIZE = 15;

    private boolean rollOver;

    /**
     * Constructor.
     */
    public CloseButton() {
      setUI(new BasicButtonUI());
      setPreferredSize(new Dimension(SIZE, SIZE));
      setToolTipText(I18n.getString("btCloseTab" + I18n.SUFFIX_TOOLTIP));
      setFocusable(false);
      setBorder(null);
      setContentAreaFilled(false);
      addMouseListener(ROLLOVER_LISTENER);
    }

    /**
     * Set whether button is under the mouse.
     * @param rollOver {@code true} if button is under mouse
     */
    public void setRollOver(boolean rollOver)
    {
      if (rollOver != this.rollOver) {
        this.rollOver = rollOver;
        repaint();
      }
    }

    /**
     * Calls the UI delegate's paint method, if the UI delegate
     * is non-{@code null}.  We pass the delegate a copy of the
     * {@code Graphics} object to protect the rest of the
     * paint code from irrevocable changes
     * (for example, {@code Graphics.translate}).
     * <p>
     * If you override this in a subclass you should not make permanent
     * changes to the passed in {@code Graphics}. For example, you
     * should not alter the clip {@code Rectangle} or modify the
     * transform. If you need to do these operations you may find it
     * easier to create a new {@code Graphics} from the passed in
     * {@code Graphics} and manipulate it. Further, if you do not
     * invoker super's implementation you must honor the opaque property,
     * that is
     * if this component is opaque, you must completely fill in the background
     * in a non-opaque color. If you do not honor the opaque property you
     * will likely see visual artifacts.
     * <p>
     * The passed in {@code Graphics} object might
     * have a transform other than the identify transform
     * installed on it.  In this case, you might get
     * unexpected results if you cumulatively apply
     * another transform.
     *
     * @param g the {@code Graphics} object to protect
     * @see #paint
     * @see javax.swing.plaf.ComponentUI
     */
    @Override
    protected void paintComponent(Graphics g)
    {
      super.paintComponent(g);
      Graphics2D g2 = (Graphics2D) g.create();
      try {
        final boolean pressed = getModel().isPressed();
        final int w = getWidth();
        final int h = getHeight();
        if (rollOver) {
          g2.setColor(ACTIVE_BG_COLOR);
          g2.fillRect(BORDER, BORDER, w - BORDER - 2, h - BORDER - 2);
          g2.setColor(pressed ? DARKER_BG_COLOR : LIGHTER_BG_COLOR);
          g2.drawLine(BORDER, BORDER, w - BORDER - 1, BORDER);
          g2.drawLine(BORDER, BORDER + 1, BORDER, h - BORDER - 1);
          g2.setColor(pressed ? LIGHTER_BG_COLOR : DARKER_BG_COLOR);
          g2.drawLine(BORDER + 1, h - BORDER - 1, w - BORDER - 1, h - BORDER - 1);
          g2.drawLine(w - BORDER - 1, BORDER + 1, w - BORDER - 1, h - BORDER - 2);
          g2.setColor(ACTIVE_FG_COLOR);
        }
        else {
          g2.setColor(getForeground());
        }
        g2.setStroke(CROSS_STROKE);
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        final int size = Math.min(w, h) - 2*CROSS_INSET - 1;
        if (size > 0) {
          g2.drawLine((w - size)/2, (h - size)/2,
                      (w + size)/2, (h + size)/2);
          g2.drawLine((w + size)/2, (h - size)/2,
                      (w - size)/2, (h + size)/2);
        }
      } finally {
        g2.dispose();
      }
    }

    /**
     * Resets the UI property to a value from the current look and
     * feel.
     *
     * @see javax.swing.JComponent#updateUI
     */
    @Override
    public void updateUI()
    {
      // no update here, because with that anything could happen
    }
  }
}
