// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 *  A temporary panel which is placed at the border in a border layout.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class BorderPanel
        extends JPanel
{
  private static final long serialVersionUID = 5267317825133581881L;
  /** The title of this panel. */
  private final JLabel titleLabel;
  /** The component. */
  private JComponent component;

  /**
   *  Creates a new BorderPanel which is not user-resizable, but user-closable
   *  @param title title of the panel
   *  @param component component to display
   */
  public BorderPanel(String title, JComponent component)
  {
    super(new BorderLayout());
    Box titleBox = Box.createHorizontalBox();
    titleLabel = new JLabel(title);
    JButton closeButton = new JButton("x");
    titleBox.add(titleLabel);
    titleBox.add(Box.createHorizontalGlue());
    titleBox.add(closeButton);
    closeButton.addActionListener(e -> setVisible(false));
    titleBox.setBorder(BorderFactory.createEmptyBorder(0, 0, 4, 0));

    add(titleBox, BorderLayout.NORTH);
    setComponent(component);
  }

  /**
   *  Set the title.
   *  @param title title to set
   */
  public void setTitle(String title)
  {
    titleLabel.setText(title);
  }

  /**
   *  Get the title.
   *  @return title
   */
  public String getTitle()
  {
    return titleLabel.getText();
  }

  /**
   *  Set the component.
   *  @param component  component
   */
  public void setComponent(JComponent component)
  {
    add(component, BorderLayout.CENTER);
    this.component = component;
  }

  /**
   *  Get the displayed component.
   *  @return component
   */
  public JComponent getComponent()
  {
    return component;
  }
}
