// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2010-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;

import javax.swing.*;
import java.awt.event.ActionEvent;

/**
 * Basic class for a toggle state.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class AbstractToggleState
        extends VisibleState
{
  /**
   * Create a visible state.
   *
   * @param labelResourceKey   resource key for labels and menu items
   * @param tooltipResourceKey resource key for tooltips
   * @param activeIcon         icon to be displayed when state is active
   * @param inactiveIcon       icon to be displayed when state is inactive
   * @param disabledIcon       icon to be displayed when state is disabled
   *                           (if {@code null} the Swing standard is used)
   */
  protected AbstractToggleState(String labelResourceKey, String tooltipResourceKey,
                                ImageIcon activeIcon, ImageIcon inactiveIcon,
                                ImageIcon disabledIcon)
  {
    super(labelResourceKey, tooltipResourceKey, activeIcon, inactiveIcon, disabledIcon);
  }

  /**
   * Create a visible state with default icons. The inactive icon is the same as the
   * active and the disabled icon is the standard disabled view.
   *
   * @param labelResourceKey   resource key for labels and menu items
   * @param tooltipResourceKey resource key for tooltips
   * @param activeIcon         icon to be displayed when state is active
   */
  protected AbstractToggleState(String labelResourceKey, String tooltipResourceKey, Icon activeIcon)
  {
    super(labelResourceKey, tooltipResourceKey, activeIcon);
  }

  /**
   * Create a visible state from I18n settings.
   *
   * @param baseTag I18n base tag
   */
  protected AbstractToggleState(@NotNull String baseTag)
  {
    super(baseTag);
  }

  /**
   * An action is performed. This is just toggling the activation.
   *
   * @param e unused
   */
  @Override
  public void actionPerformed(ActionEvent e)
  {
    if (isActivated()) {
      deactivate();
    }
    else {
      activate();
    }
  }

  /**
   * Deactivate this state.
   */
  @Override
  public void deactivate()
  {
    super.deactivate();
  }

  /**
   * Is it possible to switch this state off by clicking on it with the mouse?
   *
   * @return the answer
   */
  @Override
  public boolean isToggleEnabled()
  {
    return true;
  }
}
