// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.gimmicks.swing;

import de.caff.annotation.NotNull;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.HashMap;
import java.util.Map;

/**
 *  Basic implementation of a state.
 *  It takes care of listeners and activation.
 *
 *  @author  <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class AbstractState
        implements State
{
  /** Property change support for standard handling of property changes. */
  protected final PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this);

  /** Property list. */
  protected final Map<String, Object> properties = new HashMap<>();

  /** The state manager where we are registered. */
  protected StateManager manager;

  /**
   *  Create a new abstract state.
   */
  public AbstractState()
  {
    setPropertyInternal(ACTIVATION_PROPERTY, Boolean.FALSE);
    setPropertyInternal(ENABLE_PROPERTY, Boolean.TRUE);
  }

  /**
   *  Set the state manager which handles this state.
   *  @param stateManager state manager
   */
  @Override
  public void setManager(@NotNull StateManager stateManager)
  {
    manager = stateManager;
  }

  /**
   *  Add a property change listener.
   *  @param listener listener to add
   */
  @Override
  public void addPropertyChangeListener(@NotNull PropertyChangeListener listener)
  {
    propertyChangeSupport.addPropertyChangeListener(listener);
  }

  /**
   *  Add a property change listener for a special property.
   *  @param property property identifier
   *  @param listener listener to add
   */
  @Override
  public void addPropertyChangeListener(@NotNull String property, @NotNull PropertyChangeListener listener)
  {
    propertyChangeSupport.addPropertyChangeListener(property, listener);
  }

  /**
   *  Remove a property change listener.
   *  @param listener listener to remove
   */
  @Override
  public void removePropertyChangeListener(@NotNull PropertyChangeListener listener)
  {
    propertyChangeSupport.removePropertyChangeListener(listener);
  }

  /**
   *  Remove a property change listener for a special property.
   *  @param property property identifier
   *  @param listener listener to remove
   */
  @Override
  public void removePropertyChangeListener(@NotNull String property, @NotNull PropertyChangeListener listener)
  {
    propertyChangeSupport.removePropertyChangeListener(property, listener);
  }

  /**
   *  Activate this state.
   *  @return {@code true} if the activation was possible,
   *          {@code false} otherwise
   */
  @Override
  public boolean activate()
  {
    if (!isActivated()) {
      if (manager == null  ||   manager.mayActivate(this)) {
        setPropertyInternal(ACTIVATION_PROPERTY, Boolean.TRUE);
        return true;
      }
      else {
        return false;
      }
    }
    else {
      return true;
    }
  }

  /**
   *  An action is performed. This is just handled as a call to activate().
   *  @param e unused
   */
  @Override
  public void actionPerformed(ActionEvent e)
  {
    activate();
  }

  /**
   *  Is this state activated?
   *  @return the answer
   */
  @Override
  public boolean isActivated()
  {
    return (Boolean)properties.get(ACTIVATION_PROPERTY);
  }

  /**
   *  Deactivate this state.
   */
  @Override
  public void deactivate()
  {
    if (isActivated()) {
      setPropertyInternal(ACTIVATION_PROPERTY, Boolean.FALSE);
    }
  }

  /**
   *  Enable or disable this state.
   *  @param state if {@code true} enable this state, otherwise disable it
   */
  @Override
  public void setEnabled(boolean state)
  {
    if (isEnabled() != state) {
      if (manager == null  ||  manager.mayEnable(this, state)) {
        setPropertyInternal(ENABLE_PROPERTY, state ? Boolean.TRUE : Boolean.FALSE);
      }
    }
  }

  /**
   *  Is this state enabled?
   *  @return the answer
   */
  @Override
  public boolean isEnabled()
  {
    return (Boolean)properties.get(ENABLE_PROPERTY);
  }

  /**
   *  Get the value of a given property.
   *  @param name property name
   */
  @Override
  public Object getValue(@NotNull String name)
  {
    return properties.get(name);
  }

  /**
   *  Set the value of a given property,
   *  @param key   property key
   *  @param value new value of property
   */
  @Override
  public void putValue(@NotNull String key, Object value)
  {
    Object oldValue = getValue(key);
    if (oldValue != value  ||  (oldValue != null  &&  !oldValue.equals(value))) {
      properties.put(key, value);
      propertyChangeSupport.firePropertyChange(key, oldValue, value);
    }
  }

  /**
   *  Set a property to a new value.
   *  @param name  property name
   *  @param value  new value
   */
  protected void setProperty(@NotNull String name, Object value)
  {
    Object oldValue = getValue(name);

    // handle special values
    if (ACTIVATION_PROPERTY.equals(name)  ||
        ENABLE_PROPERTY.equals(name)) {
      throw new RuntimeException("Don't set property "+name+" via setProperty(). Use dedicated method instead.");
    }
    if (oldValue != value) {
      if (value == null  ||  oldValue == null  ||  !value.equals(oldValue)) {
        properties.put(name, value);
        propertyChangeSupport.firePropertyChange(name, oldValue, value);
      }
    }
  }

  /**
   *  Set a property to a new value (internal version).
   *  @param name  property name
   *  @param value  new value
   */
  private void setPropertyInternal(String name, Object value)
  {
    Object oldValue = getValue(name);

    properties.put(name, value);
    propertyChangeSupport.firePropertyChange(name, oldValue, value);
  }

  /**
   * Show a popup menu.
   * Implementations are free to do nothing.
   *
   * @param comp component where to show the popup menu
   * @param x    x position in component
   * @param y    y position in component
   */
  @Override
  public void showPopup(@NotNull Component comp, int x, int y)
  {
    // doing nothing
  }

  /**
   * Add this element to the toolbar.
   * @param toolBar toolbar to add to
   * @return components added to toolbar
   */
  @NotNull
  @Override
  public JComponent[] addToToolBar(@NotNull JToolBar toolBar)
  {
    final StateButton comp = new StateButton(this);
    toolBar.add(comp);
    return new JComponent[] { comp };
  }

  /**
   * Add this element to the toolbar.
   * @param toolBar toolbar to add to
   * @param index   position where to add the item
   * @return actually inserted components
   */
  @NotNull
  @Override
  public JComponent[] addToToolBar(@NotNull JToolBar toolBar, int index)
  {
    final StateButton comp = new StateButton(this);
    toolBar.add(comp, null, index);
    return new JComponent[] { comp };
  }

  /**
   * Is it possible to switch this state off by clicking on it with the mouse?
   *
   * @return the answer
   */
  @Override
  public boolean isToggleEnabled()
  {
    return false;  
  }

  /**
   * Refresh state.
   */
  @Override
  public void refresh()
  {
  }
}
